<?php

namespace App\Sys\Services\FormGroupArrange;

use App\Sys\Repository\FormGroupArrange\FormGroupArrangeRepository;
use App\Sys\Services;
use Illuminate\Support\Facades\Validator;

class FormGroupArrangeService extends Services
{
    private $arrangeRepo;

    public function __construct()
    {
        $this->arrangeRepo = new FormGroupArrangeRepository();
    }

    // Get all arrangements for a given form group
    public function getByGroup($groupId)
    {
        return $this->arrangeRepo->getByGroup($groupId);
    }

    // Sync arrangements for a form group (bulk creation)
    // This method validates the incoming array of arrangements and then calls the repository's syncGroupArrangements method.
    // The repository method is responsible for creating new arrangement records and updating the corresponding
    // order_of_cycle and order_of_arrangement columns in the forms table.
    public function syncArrangements($formGroupId, array $arrangements)
    {
        $rules = [
            '*.form_id'               => 'required|integer|exists:forms,id',
            '*.order_of_cycle'        => 'required|integer|min:1',
            '*.order_of_arrangement'  => 'required|integer|min:1',
        ];

        $validator = Validator::make($arrangements, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());
            return false;
        }

        // Check for duplicate positions
        $positions = [];
        foreach ($arrangements as $arrangement) {
            $posKey = $arrangement['order_of_cycle'] . '-' . $arrangement['order_of_arrangement'];
            if (isset($positions[$posKey])) {
                $this->setError('Duplicate position detected for cycle ' . $arrangement['order_of_cycle'] . ' arrangement ' . $arrangement['order_of_arrangement']);
                return false;
            }
            $positions[$posKey] = true;
        }

        return $this->arrangeRepo->syncGroupArrangements($formGroupId, $arrangements);
    }

    // Update arrangements for a form group (bulk update)
    // This method validates an array of arrangement updates and then calls the repository method updateGroupArrangements.
    // The repository method will update each arrangement record and also update the corresponding columns in the forms table.

    public function updateArrangements($formGroupId, array $arrangements)
    {
        $rules = [
            '*.form_id'              => 'required|integer|exists:forms,id|exists:form_group_arranges,form_id,form_group_id,' . $formGroupId,
            '*.order_of_cycle'       => 'required|integer|min:1',
            '*.order_of_arrangement' => 'required|integer|min:1',
        ];

        $validator = Validator::make($arrangements, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());
            return false;
        }

        // Check for duplicate positions
        $positions = [];
        foreach ($arrangements as $arrangement) {
            $posKey = $arrangement['order_of_cycle'] . '-' . $arrangement['order_of_arrangement'];
            if (isset($positions[$posKey])) {
                $this->setError('Duplicate position detected for cycle ' . $arrangement['order_of_cycle'] . ' arrangement ' . $arrangement['order_of_arrangement']);
                return false;
            }
            $positions[$posKey] = true;
        }

        return $this->arrangeRepo->updateGroupArrangements($formGroupId, $arrangements);
    }

    // Single arrangement add (if needed)
    public function add($data)
    {
        $rules = [
            'form_id'               => 'required|integer|exists:forms,id',
            'form_group_id'         => 'required|integer|exists:form_groups,id',
            'order_of_cycle'        => 'nullable|integer|min:1',
            'order_of_arrangement'  => 'nullable|integer|min:1',
        ];

        $valid = Validator::make($data, $rules);
        if ($valid->fails()) {
            $this->setError($valid->errors());
            return false;
        }

        $result = $this->arrangeRepo->create($data);
        // After creating a single arrangement, update the corresponding form record as well.
        // (This logic can also be handled in the repository.)
        if ($result) {
            \App\Models\Form::where('id', $data['form_id'])->update([
                'order_of_cycle' => $data['order_of_cycle'],
                'order_of_arrangement' => $data['order_of_arrangement'],
                'form_group_id' => $data['form_group_id'],
            ]);
        }
        return $result;
    }

    // Single arrangement update (if needed)
    public function update($data)
    {
        $rules = [
            'id'                    => 'required|integer',
            'form_id'               => 'sometimes|required|integer|exists:forms,id',
            'form_group_id'         => 'sometimes|required|integer|exists:form_groups,id',
            'order_of_cycle'        => 'sometimes|nullable|integer|min:1',
            'order_of_arrangement'  => 'sometimes|nullable|integer|min:1',
        ];

        $valid = Validator::make($data, $rules);
        if ($valid->fails()) {
            $this->setError($valid->errors());
            return false;
        }

        $result = $this->arrangeRepo->update($data['id'], $data);
        // Update the corresponding form record with new ordering values.
        if ($result && isset($data['form_id'])) {
            \App\Models\Form::where('id', $data['form_id'])->update([
                'order_of_cycle' => $data['order_of_cycle'],
                'order_of_arrangement' => $data['order_of_arrangement'],
                'form_group_id' => $data['form_group_id'],
            ]);
        }
        return $result;
    }

    // Other standard methods:
    public function get()
    {
        return $this->arrangeRepo->all();
    }

    public function getById($id)
    {
        try {
            return $this->arrangeRepo->find($id);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            $this->setError('Record not found');
            return false;
        }
    }

    public function del($id)
    {
        try {
            return $this->arrangeRepo->delete($id);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            $this->setError('Record not found');
            return false;
        }
    }
}
