<?php

namespace App\Sys\Services\Form;

use App\Models\ProjectForm;
use App\Sys\Repository\FormRepo;
use App\Sys\Repository\Forms\FormBranchProjectRepo;
use Carbon\Carbon;

class test
{
    protected $formRepo;
    private $fBPRepo;

    public function __construct()
    {
        $this->formRepo = new FormRepo();
        $this->fBPRepo = new FormBranchProjectRepo();
    }

    public function store($data)
    {
        // Validate data
        if (!$this->validateData($data)) {
            return false;
        }

        // Store the form
        $form = $this->formRepo->storeFrom($data);

        // Store rows and their relationships
        $this->storeRows($form, $data['rows'] ?? []);
        $this->formUpdateNext($form);
        // add file

        $this->addfiles($data, $form->id);
        return $form;
    }

    public function continue($data)
    {
        $rules = [
            'id' => 'required|exists:forms,id',
            'branches' => 'required'
        ];

        $validator = validator($data, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());
            return false;
        }


        $form = $this->formRepo->getFormById($data['id']);
        $this->formBranchProject($form, $data);
        $this->formRepo->updatedStep($data['id']);
        if ($form->next_send_date == date('Y-m-d')) {
            $projectFrom = new Services\ProjectsFroms\ProjectsFormServices();
            $projectFrom->addProjectFrom(collect($form)->toArray());
            $this->formRepo->updateFormNext($data['id'], null);
        }


        return true;
    }

    private function addfiles($data, $form_id)
    {
        if (!empty($data['files'])) {
            foreach ($data['files'] as $file) {
                $name = Services\UploadFile::uploadFile($file['value'], $file['file_type']);
                $type = $file['file_type'] == "application/pdf" ? "file" : "photo";
                $this->formRepo->addFormFile($form_id, $name, $type,$file['file_name']);
            }
        }
    }

    private function formBranchProject($form, $data)
    {
        if ($data['branches'] == 0) {
            // all bra
            $this->allProject($form->id);
        } else {
            foreach ($data['branches'] as $key => $project) {
                $bracnh = in_array(0, $project) ? $key : null;
                $this->periodicProject($form->id, $project, $bracnh);
            }
        }
    }

    private function allProject($id)
    {
        $project = $this->fBPRepo->allProject();
        foreach ($project as $pro) {
            $this->fBPRepo->add(['id' => $id, 'branch' => $pro->branch_id, 'project' => $pro->id]);
        }
    }

    public function periodicProject($id, $projectsId, $bracnh = null)
    {
        $project = $bracnh == null ? $this->fBPRepo->projects($projectsId) : $this->fBPRepo->projectsByBranch($bracnh);
        foreach ($project as $pro) {
            $this->fBPRepo->add(['id' => $id, 'branch' => $pro->branch_id, 'project' => $pro->id]);
        }
    }

    /**
     * Validate the incoming data.
     */
    private function validateData($data): bool
    {
        $rules = [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|in:general,fixed',
            'periodic_oneTime' => 'required_if:type,general|in:oneTime,periodic',
            'periodic_type' => 'nullable|in:daily,weekly,monthly,quarterly',
            'deadline_by_num_of_days' => 'nullable|numeric|min:1',
            'week_day' => 'nullable|numeric|between:1,7',
            'month_day' => 'nullable|numeric|between:1,31',
            'quarter_start_day' => 'nullable|date',
            'next_send_date' => 'nullable|date',
            'approved' => 'nullable|boolean',
            'title_print' => 'nullable|string|max:255',
            'footer_print' => 'nullable|string|max:255',
            'chart' => 'nullable|boolean',

            'files' => 'nullable|array',
            'files.*.file_type' => 'required|string|in:image/jpeg,image/jpg,image/png,application/pdf',
            'rows' => 'required|array|min:1',
            'rows.*.single_multi_column' => 'required|in:single_column,multi_column',
            'rows.*.single_multi_input' => 'required|in:single_input,multi_input',

            'rows.*.columns' => 'required|array|min:1',
            'rows.*.columns.*.width' => 'required|numeric|min:1|max:100',

            'rows.*.columns.*.inputs' => 'required|array|min:1',
            'rows.*.columns.*.inputs.*.label' => 'required|string|max:255',
            'rows.*.columns.*.inputs.*.input_type' => 'required|in:file,text,textarea,number,checkbox,radio,label,date,time,datetime',
            'rows.*.columns.*.inputs.*.placeholder' => 'nullable|string|max:255',
            'rows.*.columns.*.inputs.*.value' => 'nullable|string',
            'rows.*.columns.*.inputs.*.required' => 'required|numeric|in:0,1',
            'rows.*.columns.*.inputs.*.chart' => 'nullable|numeric|in:0,1',
            // 'rows.*.columns.*.inputs.*.chart_type' => 'nullable|in:fixed,dynamic',
            'rows.*.columns.*.inputs.*.icon' => 'nullable|string|max:255',

            'rows.*.columns.*.inputs.*.options' => 'nullable|array',
            'rows.*.columns.*.inputs.*.options.*.name' => 'required|string|max:255',

        ];

        $validator = validator($data, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());
            return false;
        }

        return true;
    }

    /**
     * Store rows and their nested relationships.
     */
    private function storeRows($form, array $rows): void
    {
        foreach ($rows as $rowData) {
            $row = $this->formRepo->storeRow($form, $rowData);
            $this->storeColumns($row, $rowData['columns'] ?? [], $form);
        }
    }

    /**
     * Store columns and their nested relationships.
     */
    private function storeColumns($row, array $columns, $form): void
    {
        foreach ($columns as $columnData) {
            $column = $this->formRepo->storeColumn($row, $columnData);
            if(isset($columnData['rows'])){
                // sub_colume
            }else{
                $this->storeInputs($column, $columnData['inputs'] ?? [], $form, $row);
            }

        }
    }

    /**
     * Store inputs and their options.
     */
    private function storeInputs($column, array $inputs, $form, $row): void
    {
        foreach ($inputs as $inputData) {
            $input = $this->formRepo->storeInput($form, $column, $inputData, $row);
            $this->storeInputOptions($input, $inputData['options'] ?? []);
        }
    }

    /**
     * Store input options.
     */
    private function storeInputOptions($input, array $options): void
    {
        foreach ($options as $optionData) {
            $this->formRepo->storeInputOption($input, $optionData);
        }
    }

    private function formUpdateNext($form): void
    {
        if ($form->type === 'general' && $form->periodic_oneTime === 'periodic') {
            switch ($form->periodic_type) {
                case 'weekly':
                    $now = Carbon::now();
                    $dayOfWeek = $now->dayOfWeek;
                    $N = ($form->week_day == 7) ? 0 : $form->week_day;
                    $dayDifference = ($N - $dayOfWeek + 7) % 7;

                    // Move to next week if today is the scheduled day
                    $nextDate = $now->copy()->addDays($dayDifference ?: 7);

                    $this->formRepo->updateFormNext($form->id, $nextDate);
                    break;

                case 'monthly':
                    $now = Carbon::now();
                    $dayOfMonth = $form->month_day;
                    // Set the next send date to the same day in the next month
                    $nextDate = $now->copy()->addMonth()->day($dayOfMonth);
                    $this->formRepo->updateFormNext($form->id, $nextDate);
                    break;

                case 'quarterly':
                    $now = Carbon::now();
                    $startDay = $form->quarter_start_day;
                    // Set the next quarter date
                    $nextDate = Carbon::parse($startDay)->addMonths(3);
                    $this->formRepo->updateFormNext($form->id, $nextDate);
                    break;

                default:
                    // No periodic update needed
                    break;
            }
        }

        if ($form->type === 'general' && $form->periodic_oneTime === 'oneTime') {
            $nextDate = $form->onetime_send == null ? Carbon::now()->format('Y-m-d') : $form->onetime_send;
            $this->formRepo->updateFormNext($form->id, $nextDate);
        }
    }

    public function updatedNextDate($form)
    {
        $nextDate = null;
        if ($form['periodic_oneTime'] != "oneTime") {
            switch ($form['periodic_type']) {
                case 'week':
                    $nextDate = date('Y-m-d', strtotime('+7 day'));
                    break;
                case 'month':
                    $nextDate = date('Y-m-d', strtotime('+31 day'));
                    break;
                case 'quarter':
                    $nextDate = date('Y-m-d', strtotime('+91 day'));
                    break;
                default:
                    # code...
                    break;
            }
        }
        $this->formRepo->updateFormNext($form['id'], $nextDate);
    }

    public
    function FromByNextDate($next)
    {
        return $this->formRepo->FromByNextDate($next);

    }

    public function FormByType()
    {
        return $this->formRepo->getFormByType();
    }


    public function update($id, $data)
    {
        if (!$this->validateUpdateData($data)) {
            return false;
        }

        // Update form basic data
        $form = $this->formRepo->updateForm($id, $data);

        if (!$form) {
            $this->setError("Form Not found");
        }

        if (!empty($data['inputs'])) {
            foreach ($data['inputs'] as $inputId => $inputData) {
                $input = $this->formRepo->updateFormInput($inputId, $inputData);
                if (!$input) {
                    $this->setError("Input $inputId not found");
                }
            }
        }
        return $form;
    }

    private function validateUpdateData($data): bool
    {
        $rules = [
            // Update basic information about the form
            'name' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'chart' => 'nullable|boolean',
            'title_print' => 'nullable|string|max:255',
            'footer_print' => 'nullable|string|max:255',
            'is_active' => 'nullable|boolean',
            'approved' => 'nullable|boolean',

            // Update in form inputs
            'inputs' => 'nullable|array',
            'inputs.*.label' => 'nullable|string|max:255',
            'inputs.*.placeholder' => 'nullable|string|max:255',
            'inputs.*.chart_type' => 'nullable|in:fixed,chart',
            'inputs.*.required' => 'nullable|numeric|in:0,1',
        ];

        $validator = validator($data, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());
            return false;
        }

        return true;
    }

    /**
     * Generate a PDF report for a specific project form.
     *
     * @param int $projectFormId The ID of the project form to generate the PDF for
     * @return \Illuminate\Http\Response The PDF response for download
     */
    public function generatePdf($projectFormId)
    {
        // Retrieve form data and project inputs
        $formData = $this->prepareFormData($projectFormId);

        // Configure and generate the PDF
        $pdf = $this->createPdf($formData);

//        $form = $formData['form'];
        $formId = $formData['id'];
        $dateOfGeneration = now()->format('Y-m-d');
        $filename = "hse_report_{$formId}_{$dateOfGeneration}.pdf";
        $storagePath = public_path("form/{$formId}/{$projectFormId}/{$filename}");

        if(!file_exists($storagePath)){
            mkdir(dirname($storagePath), 0777, true);
        }

        $pdf->save($storagePath);

        ProjectForm::where('id', $projectFormId)->update([
            'last_generated_report' => $filename,
        ]);

        $url = url("form/{$formId}/{$projectFormId}/{$filename}");

        return $url;

//        // Return the PDF for download
//        return $pdf->download('safety_inspection_report.pdf');
    }

    /**
     * Prepare and transform form data, including answers, for PDF generation.
     *
     * @param int $projectFormId The ID of the project form
     * @return array Transformed form data ready for PDF rendering
     */
    private function prepareFormData($projectFormId)
    {
        // Fetch form and project inputs from the Repo
        $data = $this->formRepo->getFullFormWithAnswersByProjectFormId($projectFormId);
        $form = $data['form'];
        $projectInputs = $data['projectInputs'];

        // Log form data for debugging
        \Log::info('Form Data: ' . json_encode($form));
        \Log::info('Project Inputs: ' . json_encode($projectInputs));

        // Transform the form data into a structure suitable for the PDF template
        return [
            'id' => $form->id,
            'name' => $form->name,
            'description' => $form->description,
            'type' => $form->type,
            'periodic_oneTime' => $form->periodic_oneTime,
            'periodic_type' => $form->periodic_type,
            'title_print' => $form->title_print ?? "تقرير السلامة - " . date('Y-m-d'),
            'footer_print' => $form->footer_print ?? "تم التحضير بواسطة قسم السلامة",
            'approved' => $form->approved,
            'chart' => $form->chart,
            'deadline_by_num_of_days' => $form->deadline_by_num_of_days,
            'week_day' => $form->week_day,
            'month_day' => $form->month_day,
            'issue_date' => $form->issue_date ?? '01/02/2025',
            'revision_date' => $form->revision_date ?? '—',
            'rows' => $this->transformRows($form->rows, $projectInputs),
        ];
    }

    /**
     * Transform form rows into grouped row structures for PDF rendering.
     *
     * @param \Illuminate\Support\Collection $rows The collection of form rows
     * @param \Illuminate\Support\Collection $projectInputs The project inputs with answers
     * @return array Transformed rows with grouped inputs
     */
    private function transformRows($rows, $projectInputs)
    {
        return $rows->map(function ($row) use ($projectInputs) {
            return $this->groupRowInputs($row, $projectInputs);
        })->flatten(1)->all();
    }

    /**
     * Group inputs within a row into pairs and create row groups.
     *
     * @param object $row The form row object
     * @param \Illuminate\Support\Collection $projectInputs The project inputs with answers
     * @return array Grouped row data for PDF rendering
     */
    private function groupRowInputs($row, $projectInputs)
    {
        $rowGroups = [];

        // Process each column to group inputs
        $columns = $row->columns->map(function ($column) use ($projectInputs) {
            return $this->processColumnInputs($column, $projectInputs);
        })->all();

        // Determine the maximum number of input groups across all columns
        $maxPairs = max(array_map(fn($column) => count($column['inputs'] ?? []), $columns));

        // Create row groups based on the maximum number of pairs
        for ($i = 0; $i < $maxPairs; $i++) {
            $rowGroups[] = [
                'single_multi_column' => $row->single_multi_column,
                'single_multi_input' => $row->single_multi_input,
                'type_row' => $row->type_row,
                'columns' => array_map(function ($column) use ($i) {
                    return [
                        'width' => $column['width'],
                        'inputs' => $column['inputs'][$i] ?? [],
                    ];
                }, $columns),
            ];
        }

        return $rowGroups;
    }

    /**
     * Process and group inputs for a specific column.
     *
     * @param object $column The form column object
     * @param \Illuminate\Support\Collection $projectInputs The project inputs with answers
     * @return array Processed column data with grouped inputs
     */
    private function processColumnInputs($column, $projectInputs)
    {
        // Count total project inputs for dynamic grouping (if needed)
        $columnData = [
            'form_column_count' => count($projectInputs),
        ];

        // Transform inputs with their answers
        $inputs = $column->inputs->map(function ($input) use ($projectInputs) {
            return $this->formatInputData($input, $projectInputs);
        })->all();

        // Group inputs into chunks based on the column count (or default to pairs if not specified)
        $groupedInputs = [];
        $chunkSize = (int)$columnData['form_column_count'] ?: 2; // Default to 2 if no specific count
        $chunks = array_chunk($inputs, $chunkSize);

        foreach ($chunks as $chunk) {
            $groupedInputs[] = $chunk;
        }

        return [
            'width' => $column->width,
            'inputs' => $groupedInputs,
        ];
    }

    /**
     * Format individual input data with answers and options.
     *
     * @param object $input The form input object
     * @param \Illuminate\Support\Collection $projectInputs The project inputs with answers
     * @return array Formatted input data
     */
    private function formatInputData($input, $projectInputs)
    {
        $projectInput = $projectInputs->get($input->id);
        $inputData = [
            'label' => $input->label ?? '',
            'form_column_id' => $input->form_column_id,
            'input_type' => $input->input_type ?? 'unknown',
            'placeholder' => $input->placeholder ?? '',
            'options' => $input->options->pluck('name', 'id')->all(),
        ];

        if ($projectInput) {
            $rawAnswer = $projectInput->answer;
            if ($rawAnswer !== null) {
                if (is_string($rawAnswer) && (strpos($rawAnswer, '[') === 0 || strpos($rawAnswer, '"') !== false)) {
                    $answers = json_decode($rawAnswer, true);
                    if (json_last_error() === JSON_ERROR_NONE && is_array($answers)) {
                        $formattedAnswers = [];
                        foreach ($answers as $optionId) {
                            if (isset($inputData['options'][$optionId])) {
                                $formattedAnswers[] = $inputData['options'][$optionId] ?: $optionId;
                            } else {
                                $formattedAnswers[] = $optionId;
                            }
                        }
                        $inputData['answer'] = !empty($formattedAnswers) ? implode(', ', $formattedAnswers) : 'No answer selected';
                    } else {
                        $answers = array_filter(explode(',', trim($rawAnswer)));
                        $formattedAnswers = array_map('trim', $answers);
                        $inputData['answer'] = !empty($formattedAnswers) ? implode(', ', $formattedAnswers) : 'No answer selected';
                    }
                } else {
                    $inputData['answer'] = trim($rawAnswer) ?: 'No answer selected';
                }
            } else {
                $inputData['answer'] = 'No answer selected';
            }
        } else {
            $inputData['answer'] = $input->value ?? $input->placeholder ?? 'No answer selected';
        }

        return $inputData;
    }

    /**
     * Create and configure the PDF document with the form data.
     *
     * @param array $formData The transformed form data for PDF rendering
     * @return \Mccarlosen\LaravelMpdf\LaravelMpdf The configured PDF instance
     */
    private function createPdf($formData)
    {
        return PDF::loadView('pdf.form_report', ['form' => $formData]);
        //return view('pdf.form_report')->with('form',$formData);
    }
}
