<?php

namespace App\Sys\Services\Form;

use App\Http\Requests\FormCreateRequest;
use App\Models\Branch;
use App\Models\Form;
use App\Models\ProjectForm;
use App\Models\ProjectInput;
use App\Sys\Repository\formComplexGroupsRepo;
use App\Sys\Repository\FormRepo;
use App\Sys\Repository\Forms\FormBranchProjectRepo;
use App\Sys\Services;
use Carbon\Carbon;
use Illuminate\Http\Request;
use validation;
use PDF;
use function Symfony\Component\Translation\t;


class FormService extends Services
{
    protected $formRepo;
    private $fBPRepo;
    private $formComplexGroup;

    public function __construct()
    {
        $this->formRepo = new FormRepo();
        $this->fBPRepo = new FormBranchProjectRepo();
        $this->formComplexGroup = new formComplexGroupsRepo();
    }

    public function getAllForms($types)
    {
        return $this->formRepo->getAll($types);
    }

    public function getByBranch($id)
    {
        return $this->formRepo->getByBranch($id);
    }


    public function updatedForm($data)
    {
        $rules = [
            'id' => 'required|exists:forms,id',
        ];
        $validator = validator($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());
            return false;
        }

        if ($form = $this->store($data)) {
            $this->disableForms($data['id']);
            return $form;
        }

        return false;
    }

    public function disableForms($id)
    {
        if ($this->formRepo->disable($id))
            return true;

        return false;
    }

    public function store($data)
    {
        // Ensure that only authenticated users can create a form.


        // This will use the currently authenticated user if available.
        if (!isset($data['user_id'])) {
            $data['user_id'] = auth()->id();
        }

        // Validate data
        if (!$this->validateData($data)) {
            return false;
        }
        // Store the form
        $form = $this->formRepo->storeFrom($data);

        // Store rows and their relationships
        $this->storeRows($form, $data['rows'] ?? []);
        $this->formUpdateNext($form);
        // add file

        $this->addfiles($data, $form->id);
        return $form;
    }

    public function continue($data)
    {
        $rules = [
            'id' => 'required|exists:forms,id',
            'branches' => 'required'
        ];

        $validator = validator($data, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());
            return false;
        }


        $form = $this->formRepo->getFormById($data['id']);
        $this->formBranchProject($form, $data);
        $this->formRepo->updatedStep($data['id']);
        if ($form->next_send_date == date('Y-m-d')) {
            $projectFrom = new Services\ProjectsFroms\ProjectsFormServices();
            $projectFrom->addProjectFrom(collect($form)->toArray());
            $this->formRepo->updateFormNext($data['id'], null);
        }


        return true;
    }

    private function addfiles($data, $form_id)
    {
        if (!empty($data['files'])) {
            foreach ($data['files'] as $file) {
                $name = Services\UploadFile::uploadFile($file['value'], $file['file_type']);
                $type = $file['file_type'] == "application/pdf" ? "file" : "photo";
                $this->formRepo->addFormFile($form_id, $name, $type, $file['file_name']);
            }
        }
    }

    private function formBranchProject($form, $data)
    {
        if ($data['branches'] == 0) {
            // all bra
            $this->allProject($form->id);
        } else {
            foreach ($data['branches'] as $key => $project) {
                $bracnh = in_array(0, $project) ? $key : null;
                $this->periodicProject($form->id, $project, $bracnh);
            }
        }
    }

    private function allProject($id)
    {
        $project = $this->fBPRepo->allProject();
        foreach ($project as $pro) {
            $this->fBPRepo->add(['id' => $id, 'branch' => $pro->branch_id, 'project' => $pro->id]);
        }
    }

    public function periodicProject($id, $projectsId, $bracnh = null)
    {
        $project = $bracnh == null ? $this->fBPRepo->projects($projectsId) : $this->fBPRepo->projectsByBranch($bracnh);
        foreach ($project as $pro) {
            $this->fBPRepo->add(['id' => $id, 'branch' => $pro->branch_id, 'project' => $pro->id]);
        }
    }

    /**
     * Validate the incoming data.
     */
    private function validateData($data): bool
    {
        $rules = [
            'name' => 'required|string|max:255',
            'form_group_id' => 'sometimes|exists:form_groups,id',
            'description' => 'nullable|string',
            'type' => 'required|in:general,fixed,complex',
            'periodic_oneTime' => 'required_if:type,general|in:oneTime,periodic',
            'periodic_type' => 'nullable|in:day,weekly,monthly,quarterly',
            'deadline_by_num_of_days' => 'nullable|numeric|min:1',
            'week_day' => 'nullable|numeric|between:1,7',
            'month_day' => 'nullable|numeric|between:1,31',
            'quarter_start_day' => 'nullable|date',
            'next_send_date' => 'nullable|date',
            'approved' => 'nullable|boolean',
            'title_print' => 'nullable|string|max:255',
            'footer_print' => 'nullable|string|max:255',
            'chart' => 'nullable|boolean',
            'order_of_cycle' => 'nullable|numeric',
            'order_of_arrangement' => 'nullable|numeric',

            'files' => 'nullable|array',
            'files.*.file_type' => 'required|string|in:image/jpeg,image/jpg,image/png,application/pdf',
            'rows' => 'required|array|min:1',
            'rows.*.single_multi_column' => 'required|in:single_column,multi_column',
            'rows.*.single_multi_input' => 'required|in:single_input,multi_input',

            'rows.*.columns' => 'required|array|min:1',
            'rows.*.columns.*.width' => 'required|numeric|min:1|max:100',

            'rows.*.columns.*.inputs' => [
                'array',
                function ($attribute, $value, $fail) use ($data) {
                    $keys = explode('.', $attribute);
                    $rowIndex = $keys[1] ?? null;
                    $columnIndex = $keys[3] ?? null;

                    if ($rowIndex !== null && $columnIndex !== null) {
                        $columnType = data_get($data, "rows.$rowIndex.columns.$columnIndex.column_type");

                        if ($columnType === 'input' && empty($value)) {
                            $fail('حقل "inputs" مطلوب عندما يكون column_type = input.');
                        }
                    }
                },
            ],
            //'rows.*.columns.*.inputs' => 'required|array|min:1',
            'rows.*.columns.*.inputs.*.label' => 'required|string|max:255',
            'rows.*.columns.*.inputs.*.input_type' => 'required|in:file,text,textarea,number,checkbox,radio,label,date,time,datetime,creator,approver',
            'rows.*.columns.*.inputs.*.placeholder' => 'nullable|string|max:255',
            'rows.*.columns.*.inputs.*.value' => 'nullable',
            'rows.*.columns.*.inputs.*.required' => 'required|numeric|in:0,1',
            // 'rows.*.columns.*.inputs.*.chart' => 'nullable|numeric|in:0,1',
            // 'rows.*.columns.*.inputs.*.chart_type' => 'nullable|in:fixed,dynamic',
            'rows.*.columns.*.inputs.*.icon' => 'nullable|string|max:255',

            'rows.*.columns.*.inputs.*.options' => 'nullable|array',
            'rows.*.columns.*.inputs.*.options.*.name' => 'required|string|max:255',

            'rows.*.columns.*.sub_rows' => 'array',
            'rows.*.columns.*.sub_rows.*.sub_columns' => 'array',
            'rows.*.columns.*.sub_rows.*.sub_columns.*.inputs' => [
                'array',
                function ($attribute, $value, $fail) use ($data) {
                    if (empty($value)) {
                        $fail('حقل "inputs" داخل sub_columns مطلوب.');
                    }
                },
            ],
            'rows.*.columns.*.sub_rows.*.sub_columns.*.inputs.*.label' => 'required|string|max:255',
            'rows.*.columns.*.sub_rows.*.sub_columns.*.inputs.*.input_type' => 'required|in:file,text,textarea,number,checkbox,radio,label,date,time,datetime,creator,approver',
            'rows.*.columns.*.sub_rows.*.sub_columns.*.inputs.*.placeholder' => 'nullable|string|max:255',
            'rows.*.columns.*.sub_rows.*.sub_columns.*.inputs.*.value' => 'nullable',
            'rows.*.columns.*.sub_rows.*.sub_columns.*.inputs.*.required' => 'required|numeric|in:0,1',
            'rows.*.columns.*.sub_rows.*.sub_columns.*.inputs.*.show_pdf' => 'required|numeric|in:0,1',
            // 'rows.*.columns.*.sub_rows.*.sub_columns.*.inputs.*.chart' => 'nullable|numeric|in:0,1',
            // 'rows.*.columns.*.inputs.*.chart_type' => 'nullable|in:fixed,dynamic',
            'rows.*.columns.*.sub_rows.*.sub_columns.*.inputs.*.icon' => 'nullable|string|max:255',

            'rows.*.columns.*.sub_rows.*.sub_columns.*.inputs.*.options' => 'nullable|array',
            'rows.*.columns.*.sub_rows.*.sub_columns.*.inputs.*.options.*.name' => 'required|string|max:255',


        ];

        $validator = validator($data, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());
            return false;
        }

        return true;
    }

    /**
     * Store rows and their nested relationships.
     */
    private function storeRows($form, array $rows): void
    {
        foreach ($rows as $rowData) {
            $row = $this->formRepo->storeRow($form, $rowData);
            $this->storeColumns($row, $rowData['columns'] ?? [], $form);
        }
    }

    /**
     * Store columns and their nested relationships.
     */
    private function storeColumns($row, array $columns, $form): void
    {
        foreach ($columns as $columnData) {
            $column = $this->formRepo->storeColumn($row, $columnData);
            if ($columnData['column_type'] == "sub") {
                $this->storeSubRows($column, $columnData['sub_rows'] ?? [], $form, $row);
            } else {
                $this->storeInputs($column, $columnData['inputs'] ?? [], $form, $row, null, null);
            }
        }
    }

    /**
     * Store inputs and their options.
     */

    private function storeSubRows($column, array $subRow, $form, $row)
    {
        foreach ($subRow as $rowData) {
            $addSubRow = $this->formRepo->storeSubRow($column, $rowData);
            $this->storeSubCloumes($addSubRow, $column, $rowData['sub_columns'], $form, $row);
        }
    }

    private function storeSubCloumes($subRow, $formColumn, $columnsData, $form, $row)
    {
        foreach ($columnsData as $column) {
            $subColumn = $this->formRepo->storeSubCloum($formColumn, $subRow, $column);
            $this->storeInputs($formColumn, $column['inputs'], $form, $row, $subRow->id, $subColumn->id);
        }
    }

    private function storeInputs($column, array $inputs, $form, $row, $subRow = null, $subColumn = null): void
    {
        foreach ($inputs as $inputData) {
            $input = $this->formRepo->storeInput($form, $column, $inputData, $row, $subRow, $subColumn);
            $this->storeInputOptions($input, $inputData['options'] ?? []);
        }
    }


    /**
     * Store input options.
     */
    private function storeInputOptions($input, array $options): void
    {
        foreach ($options as $optionData) {
            $this->formRepo->storeInputOption($input, $optionData);
        }
    }

    private function formUpdateNext($form): void
    {
        if ($form->type === 'general' && $form->periodic_oneTime === 'periodic') {
            switch ($form->periodic_type) {
                case 'weekly':
                    $now = Carbon::now();
                    $dayOfWeek = $now->dayOfWeek;
                    $N = ($form->week_day == 7) ? 0 : $form->week_day;
                    $dayDifference = ($N - $dayOfWeek + 7) % 7;

                    // Move to next week if today is the scheduled day
                    $nextDate = $now->copy()->addDays($dayDifference ?: 7);

                    $this->formRepo->updateFormNext($form->id, $nextDate);
                    break;

                case 'monthly':
                    $now = Carbon::now();
                    $dayOfMonth = $form->month_day;
                    // Set the next send date to the same day in the next month
                    $nextDate = $now->copy()->addMonth()->day($dayOfMonth);
                    $this->formRepo->updateFormNext($form->id, $nextDate);
                    break;

                case 'quarterly':
                    $now = Carbon::now();
                    $startDay = $form->quarter_start_day;
                    // Set the next quarter date
                    $nextDate = Carbon::parse($startDay)->addMonths(3);
                    $this->formRepo->updateFormNext($form->id, $nextDate);
                    break;

                default:
                    // No periodic update needed
                    break;
            }
        }

        if ($form->type === 'general' && $form->periodic_oneTime === 'oneTime') {
            $nextDate = $form->onetime_send == null ? Carbon::now()->format('Y-m-d') : $form->onetime_send;
            $this->formRepo->updateFormNext($form->id, $nextDate);
        }
    }

    public function updatedNextDate($form)
    {
        $nextDate = null;
        if ($form['periodic_oneTime'] != "oneTime") {
            switch ($form['periodic_type']) {
                case 'week':
                    $nextDate = date('Y-m-d', strtotime('+7 day'));
                    break;
                case 'month':
                    $nextDate = date('Y-m-d', strtotime('+31 day'));
                    break;
                case 'quarter':
                    $nextDate = date('Y-m-d', strtotime('+91 day'));
                    break;
                default:
                    # code...
                    break;
            }
        }
        $this->formRepo->updateFormNext($form['id'], $nextDate);
    }

    public
    function FromByNextDate($next)
    {
        return $this->formRepo->FromByNextDate($next);
    }

    public function FormByType()
    {
        return $this->formRepo->getFormByType();
    }


    public function update($id, $data)
    {

        // Ensure that only authenticated users can create a form.

        // This will use the currently authenticated user if available.
        if (!isset($data['user_id'])) {
            $data['user_id'] = auth()->id();
        }

        if (!$this->validateUpdateData($data)) {
            return false;
        }

        // Update form basic data
        $form = $this->formRepo->updateForm($id, $data);

        if (!$form) {
            $this->setError("Form Not found");
        }

        // Only sync departments if the form type is 'complex'
        // if (
        //     isset($data['type']) && $data['type'] === 'complex' &&
        //     isset($data['departments']) && is_array($data['departments']) &&
        //     isset($data['form_group_id'])
        // ) {
        //     $pivotData = [];
        //     foreach ($data['departments'] as $departmentId) {
        //         $pivotData[$departmentId] = ['form_group_id' => $data['form_group_id']];
        //     }
        //     $form->departments()->sync($pivotData);
        // }

        if (!empty($data['inputs'])) {
            foreach ($data['inputs'] as $inputId => $inputData) {
                $input = $this->formRepo->updateFormInput($inputId, $inputData);
                if (!$input) {
                    $this->setError("Input $inputId not found");
                }
            }
        }
        return $form;
    }

    private function validateUpdateData($data): bool
    {
        $rules = [
            // Update basic information about the form
            'form_group_id' => 'sometimes|exists:form_groups,id',
            'name' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'chart' => 'nullable|boolean',
            'title_print' => 'nullable|string|max:255',
            'footer_print' => 'nullable|string|max:255',
            'is_active' => 'nullable|boolean',
            'approved' => 'nullable|boolean',
            'order_of_cycle' => 'nullable|numeric',
            'order_of_arrangement' => 'nullable|numeric',
            // Add departments validation
            // 'departments' => 'nullable|array',
            // 'departments.*' => 'exists:department,id',

            // Update in form inputs
            'inputs' => 'nullable|array',
            'inputs.*.label' => 'nullable|string|max:255',
            'inputs.*.placeholder' => 'nullable|string|max:255',
            'inputs.*.chart_type' => 'nullable|in:fixed,chart',
            'inputs.*.required' => 'nullable|numeric|in:0,1',
            'inputs.*.show_pdf' => 'nullable|numeric|in:0,1',
        ];

        $validator = validator($data, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());
            return false;
        }

        return true;
    }

    /**
     * Generate a PDF report for a specific project form.
     *
     * @param int $projectFormId The ID of the project form to generate the PDF for
     * @return \Illuminate\Http\Response The PDF response for download
     */
    public function generatePdf($projectFormId)
    {
        $formData = $this->prepareFormData($projectFormId);
        // print_r('<pre/>');
        // print_r($formData['type']);
        if ($formData['type'] == "complex") {
            $complexData =[];
            $formsGroup = $this->formComplexGroup->getGroup($formData['form_group']);
            $formComplex = $this->formComplexGroup->getByReference($formData['reference']);
            foreach ($formComplex as $fo)
            {
                if($fo->project_form_id != null)
                    $complexData[] = $this->prepareFormData($fo->project_form_id);
            }
            //  $pdf = $this->createPdfCompalex($formsGroup,$complexData);
            return   $pdf = $this->createPdfCompalex($formsGroup,$complexData);
        }else{
            //  $pdf = $this->createPdf($formData);
            return  $pdf = $this->createPdf($formData);
        }
        $formId = $formData['id'];
        $dateOfGeneration = now()->format('Y-m-d');
        $filename = "hse_report_{$formId}_{$dateOfGeneration}.pdf";
        $storagePath = public_path("form/{$formId}/{$projectFormId}/{$filename}");
        $directoryPath = dirname($storagePath);

        if (!is_dir($directoryPath)) {
            if (file_exists($directoryPath) && !is_dir($directoryPath)) {
                unlink($directoryPath);
            }
            if (!@mkdir($directoryPath, 0777, true) && !is_dir($directoryPath)) {
                $this->setError('Failed to create directory');
                return false;
            }
        }
        $pdf->save($storagePath);
        ProjectForm::where('id', $projectFormId)->update([
            'last_generated_report' => $filename,
        ]);

        $url = url("form/{$formId}/{$projectFormId}/{$filename}");

        return $url;

        // Return the PDF for download
        return $pdf->download('safety_inspection_report.pdf');
    }

    /**
     * Prepare and transform form data, including answers, for PDF generation.
     *
     * @param int $projectFormId The ID of the project form
     * @return array Transformed form data ready for PDF rendering
     */
    private function prepareFormData($projectFormId)
    {
        // Fetch form and project inputs from the Repo
        $data = $this->formRepo->getFullFormWithAnswersByProjectFormId($projectFormId);
        $form = $data['form'];
        $projectInputs = $data['projectInputs'];
        $reference = $data['reference'];

        // Log form data for debugging
        \Log::info('Form Data: ' . json_encode($form));
        \Log::info('Project Inputs: ' . json_encode($projectInputs));

        // Transform the form data into a structure suitable for the PDF template
        return [
            'id' => $form->id,
            'name' => $form->name,
            'description' => $form->description,
            'type' => $form->type,
            'form_group'=>$form->form_group_id,
            'periodic_oneTime' => $form->periodic_oneTime,
            'periodic_type' => $form->periodic_type,
            'title_print' => $form->title_print ?? "تقرير السلامة - " . date('Y-m-d'),
            'footer_print' => $form->footer_print ?? "تم التحضير بواسطة قسم السلامة",
            'approved' => $form->approved,
            'chart' => $form->chart,
            'deadline_by_num_of_days' => $form->deadline_by_num_of_days,
            'week_day' => $form->week_day,
            'month_day' => $form->month_day,
            'issue_date' => $form->issue_date ?? date('d-m-Y'),
            'revision_date' => $form->revision_date ?? '—',
            "reference" => $reference,
            'form_group_id' => $form->form_group_id,
            'rows' => $this->transformRows($form->rows, $projectInputs),
        ];
    }

    /**
     * Transform form rows into grouped row structures for PDF rendering.
     *
     * @param \Illuminate\Support\Collection $rows The collection of form rows
     * @param \Illuminate\Support\Collection $projectInputs The project inputs with answers
     * @return array Transformed rows with grouped inputs
     */
    private function transformRows($rows, $projectInputs)
    {
        return $rows->map(function ($row) use ($projectInputs) {
            return $this->groupRowInputs($row, $projectInputs);
        })->flatten(1)->all();
    }

    /**
     * Group inputs within a row into pairs and create row groups.
     *
     * @param object $row The form row object
     * @param \Illuminate\Support\Collection $projectInputs The project inputs with answers
     * @return array Grouped row data for PDF rendering
     */
    private function groupRowInputs($row, $projectInputs)
    {
        // writing_mode
        // dd($projectInputs);
        $rowGroups = [];

        // Process each column to group inputs
        $columns = $row->columns->map(function ($column) use ($projectInputs, $row) {
            return $this->processColumnInputs($column, $projectInputs, $row);
        })->all();

        if (array_filter($columns, fn($col) => isset($col['sub_rows']))) {

            $rowGroups[] =
                [
                    'single_multi_column' => $row->single_multi_column,
                    'single_multi_input' => $row->single_multi_input,
                    'type_row' => $row->type_row,
                    'columns' => $columns
                ];
        } else {
            // Determine the maximum number of input groups across all columns
            $maxPairs = max(array_map(fn($column) => count($column['inputs'] ?? []), $columns));

            // Create row groups based on the maximum number of pairs
            for ($i = 0; $i < $maxPairs; $i++) {
                $rowGroups[] = [
                    'single_multi_column' => $row->single_multi_column,
                    'single_multi_input' => $row->single_multi_input,
                    'type_row' => $row->type_row,
                    'columns' => array_map(function ($column) use ($i) {
                        return [
                            'width' => $column['width'],
                            'type' => 'main-table',
                            'inputs' => $column['inputs'][$i] ?? [],
                        ];
                    }, $columns),
                ];
            }
        }


        return $rowGroups;
    }

    /**
     * Process and group inputs for a specific column.
     *
     * @param object $column The form column object
     * @param \Illuminate\Support\Collection $projectInputs The project inputs with answers
     * @return array Processed column data with grouped inputs
     */

    private function processColumnInputs($column, $projectInputs, $row)
    {


        if ($column->column_type == 'input') {
            // Count total project inputs for dynamic grouping (if needed)
            $columnData = [
                'form_column_count' => count($projectInputs),
            ];

            // Transform inputs with their answers
            $inputs = $column->inputs->map(function ($input) use ($projectInputs) {
                return $this->formatInputData($input, $projectInputs);
            })->all();

            // Group inputs into chunks based on the column count (or default to pairs if not specified)
            $groupedInputs = [];
            $chunkSize = (int)$columnData['form_column_count'] ?: 2; // Default to 2 if no specific count
            $chunks = array_chunk($inputs, $chunkSize);
            // dd($inputs);
            foreach ($chunks as $chunk) {
                $groupedInputs[] = $chunk;
            }

            return [
                'width' => $column->width,
                'type' => 'main-table-muilt-input',
                'inputs' => $groupedInputs,
            ];
        } else {
            $subData = [
                'width' => $column->width,
                'type' => 'sub-table',
                'sub_rows' => []
            ]; // type "sub"
            foreach ($column->getsubRows as $subRow) {
                //dd($subRow);
                $subData['sub_rows'][] = $this->processSubRows($subRow, $projectInputs);
            }

            // $subData = $this->processSubRows($column, $projectInputs);
            // dd($subData);
            return $subData;
            // return [
            //     'width' => $column->width, // Use the main column's width
            //     'sub'   => $subData,      // Array of sub-row groups
            // ];
        }
    }

    private function processSubRows($row, $projectInputs)
    {

        // Transform each column.
        $rowGroups = [];

        // Process each column to group inputs
        $columns = $row->getSubColumn->map(function ($column) use ($projectInputs, $row) {
            return $this->processSubColumn($column, $projectInputs, $row);
        })->all();

        // Determine the maximum number of input groups across all columns
        $maxPairs = max(array_map(fn($column) => count($column['inputs'] ?? []), $columns));

        // Create row groups based on the maximum number of pairs
        for ($i = 0; $i < $maxPairs; $i++) {
            $rowGroups[] = [
                'single_multi_column' => $row->single_multi_column,
                'single_multi_input' => $row->single_multi_input,
                'type_row' => $row->type_row,
                'columns' => array_map(function ($column) use ($i) {
                    return [
                        //'id'=>$column->id,
                        'width' => $column['width'],
                        'type' => 'main-table',
                        'inputs' => $column['inputs'][$i] ?? [],
                    ];
                }, $columns),
            ];
        }

        return $rowGroups;


        // Pass the current sub row to determine chunking.
        //  $subRowGroup[] = $this->processSubColumnInputs($subColumn, $projectInputs, $subRow);

    }

    private function processSubColumn($column, $projectInputs, $row)
    {
        $columnData = [
            'form_column_count' => count($projectInputs),
        ];

        // Transform inputs with their answers
        $inputs = $column->getInput->map(function ($input) use ($projectInputs) {
            return $this->formatInputData($input, $projectInputs);
        })->all();

        // Group inputs into chunks based on the column count (or default to pairs if not specified)
        $groupedInputs = [];
        $chunkSize = (int)$columnData['form_column_count'] ?: 2; // Default to 2 if no specific count
        $chunks = array_chunk($inputs, $chunkSize);

        foreach ($chunks as $chunk) {
            $groupedInputs[] = $chunk;
        }

        return [
            'id' => $column->id,
            'width' => $column->width,
            'type' => 'sub-table',
            //  'inputs' => count($groupedInputs[0]) > 0 ? array_values($groupedInputs[0]):$groupedInputs,
            'inputs' => $groupedInputs,
        ];
    }

    /**
     * Format individual input data with answers and options.
     *
     * @param object $input The form input object
     * @param \Illuminate\Support\Collection $projectInputs The project inputs with answers
     * @return array Formatted input data
     */
    private function formatInputData($input, $projectInputs)
    {


        $projectInput = $projectInputs->get($input->id);

        //      print_r('<pre/>');
        // print_r($input);
        $inputData = [
            "id" => $input->id,
            'label' => $input->label ?? '',
            'form_column_id' => $input->form_column_id,
            'input_type' => $input->input_type ?? 'unknown',
            'placeholder' => $input->placeholder ?? '',
            'writing_mode' => $input->writing_mode ?? '',
            'show_pdf' => $input->show_pdf ?? '',
            'options' => $input->options->pluck('name', 'id')->all(),
        ];

        if ($projectInput) {

            $rawAnswer = $projectInput->answer;
            $answerNamed = $projectInput->answer_named;
            if ($rawAnswer !== null) {
                //                   print_r('<pre/>');
                // print_r($projectInput);


                if (is_string($rawAnswer) && (strpos($rawAnswer, '[') === 0 || strpos($rawAnswer, '"') !== false)) {
                    $answers = json_decode($rawAnswer, true);

                    if (json_last_error() === JSON_ERROR_NONE && is_array($answers)) {
                        $formattedAnswers = [];

                        foreach ($answers as $optionId) {

                            if (isset($inputData['options'][$optionId])) {
                                // change Option to redio button
                                //  $formattedAnswers[] = $inputData['options'][$optionId] ?: $optionId;
                                $formattedAnswers[] = $optionId ?: $optionId;
                            } else {
                                $formattedAnswers[] = $optionId;
                            }
                        }
                        //Rideo Option
                        $inputData['answer'] = $answerNamed;
                        // $inputData['answer'] = !empty($formattedAnswers) ? implode(', ', $formattedAnswers) : ' ';
                    } else {
                        $answers = array_filter(explode(',', trim($rawAnswer)));
                        $formattedAnswers = array_map('trim', $answers);
                        $inputData['answer'] = !empty($formattedAnswers) ? implode(', ', $formattedAnswers) : ' ';
                    }
                } else {
                    if ($projectInput->input_type == 'checkbox' || $projectInput->input_type == 'radio') {
                        $inputData['answer'] = $answerNamed;
                    } else {
                        $inputData['answer'] = trim($rawAnswer) ?: ' ';
                    }

                    // $inputData['answer'] = trim($rawAnswer) ?: ' ';
                }
            } else {
                //answer_named
                $inputData['answer'] = ' ';
            }
        } else {
            $inputData['answer'] = $input->value ?? $input->placeholder ?? ' ';
        }
        // print_r('<pre/>');
        // if($inputData['answer']== ' '){
        //     print_r('name');
        // }else{
        //   print_r(gettype($inputData['answer']));
        // }


        return $inputData;
    }

    /**
     * Create and configure the PDF document with the form data.
     *
     * @param array $formData The transformed form data for PDF rendering
     * @return \Mccarlosen\LaravelMpdf\LaravelMpdf The configured PDF instance
     */
    private function createPdf($formData)
    {
        // return PDF::loadView('pdf.form_report', ['form' => $formData]);
        return view('pdf.form_report')->with('form',$formData);
    }

    private function createPdfCompalex($formsGroup,$complexData)
    {
        //return PDF::loadView('pdf.group', ['formsGroup' => $formsGroup,'complexData'=>$complexData]);
        return view('pdf.group',['formsGroup' => $formsGroup,'complexData'=>$complexData]);
    }

    public function getByid($id)
    {
        return $this->formRepo->getbyId($id);
    }

    public function getComaletedFroms()
    {
        $fromComplete = $this->formRepo->compalteChart();
        $fromCompleteName = [];
        $fromCompleteColor = [];
        $fromCompleteValue = [];
        foreach ($fromComplete as $form) {
            $fromCompleteName[] = $form->form_name;
            $fromCompleteValue[] = $form->complete_count;
            $fromCompleteColor[] = $this->getRandomColor();
        }
        return [
            "series" => $fromCompleteValue,
            "labels" => $fromCompleteName,
            "color" => $fromCompleteColor
        ];
    }

    public function getCompaletedNotCompalte()
    {
        $data =  $this->formRepo->chartCompalteInCompBranch();
        $colors = [];
        $categories = [];
        $complated = [];
        $inComplated = [];
        foreach ($data as $form) {
            $categories[] = $form->name;
            $colors[] = $this->getRandomColor();
            $complated[] = $form->complete_forms;
            $inComplated[] = $form->not_complete_forms;
        }
        return [
            "completed" => $complated,
            "inComplated" => $inComplated,
            "categories" => $categories,
            "color" => $colors
        ];
    }

    public function count()
    {
        return $this->formRepo->count();
    }

    private function getRandomColor()
    {
        $letters = '0123456789ABCDEF';
        $color = '#';
        for ($i = 0; $i < 6; $i++) {
            $color .= $letters[rand(0, 15)];
        }
        return $color;
    }

    public function getFromChartFixed($type='general')
    {
        return $this->formRepo->getFromChartFixed($type);
    }
    public function getFormsChartChart($type,$id)
    {
        return $this->formRepo->getFormsChartChart($type,$id);
    }

    public function getFormChartInput($id)
    {
        return $this->formRepo->getFormChartInput($id);
    }



    /**
     * Update ordering values for complex forms directly in the forms table.
     *
     * Expects an array of form data. Each item must include:
     *   - id (required)
     *   - order_of_cycle (optional)
     *   - order_of_arrangement (optional)
     *   - form_group_id (optional)
     *
     * If ordering values are missing, they are auto-assigned based on ascending order of the form id.
     * Only forms with type "complex" will be updated.
     *
     * @param array $formsData
     * @return array|false
     */

    // START ARRANGE COMPLEX FORM //

    public function updateComplexFormOrders(array $formsData)
    {
        // 1. Validate that each form exists and is of type 'complex'
        if (!$this->validateForms($formsData)) {
            return false;
        }

        // 2. Auto-assign ordering if needed
        $formsData = $this->autoAssignOrderIfNeeded($formsData);

        // 3. Check for duplicate ordering in the provided data
        if (!$this->checkForDuplicates($formsData)) {
            return false;
        }


        // 4. Check for conflicts with existing forms in the same group
        if (!$this->checkForExistingConflicts($formsData)) {
            return false;
        }

        // 5. Perform the updates
        return $this->performUpdates($formsData);
    }



    private function validateForms(array $formsData)
    {
        foreach ($formsData as $data) {
            if (!isset($data['id'])) {
                $this->setError('Each form must have an id.');
                return false;
            }
            $form = Form::find($data['id']);
            if (!$form || $form->type !== 'complex') {
                $this->setError('Form with id ' . $data['id'] . ' not found or not of type complex');
                return false;
            }
        }
        return true;
    }

    private function autoAssignOrderIfNeeded(array $formsData)
    {
        $autoOrder = false;
        foreach ($formsData as $data) {
            if (!isset($data['order_of_cycle']) || !isset($data['order_of_arrangement'])) {
                $autoOrder = true;
                break;
            }
        }
        if ($autoOrder) {
            // Sort forms by ID and auto-assign order values
            usort($formsData, function ($a, $b) {
                return $a['id'] <=> $b['id'];
            });
            $counter = 1;
            foreach ($formsData as &$data) {
                $data['order_of_cycle'] = $data['order_of_cycle'] ?? 1;
                $data['order_of_arrangement'] = $data['order_of_arrangement'] ?? $counter++;
            }
            unset($data);
        }
        return $formsData;
    }

    private function checkForDuplicates(array $formsData)
    {
        $cyclePositions = [];
        $arrangementPositions = [];

        foreach ($formsData as $data) {
            // Check order_of_cycle
            $cycle = $data['order_of_cycle'];
            if (isset($cyclePositions[$cycle])) {
                $this->setError('Duplicate cycle detected for cycle ' . $cycle);
                return false;
            }
            $cyclePositions[$cycle] = true;

            // Check order_of_arrangement
            $arrangement = $data['order_of_arrangement'];
            if (isset($arrangementPositions[$arrangement])) {
                $this->setError('Duplicate arrangement detected for arrangement ' . $arrangement);
                return false;
            }
            $arrangementPositions[$arrangement] = true;
        }

        return true;
    }


    private function checkForExistingConflicts(array $formsData)
    {
        $formGroupIds = array_unique(array_column($formsData, 'form_group_id'));
        foreach ($formGroupIds as $groupId) {
            $groupForms = array_filter($formsData, function ($form) use ($groupId) {
                return ($form['form_group_id'] ?? null) == $groupId;
            });
            $existingForms = Form::where('form_group_id', $groupId)
                ->where('type', 'complex')
                ->whereNotIn('id', array_column($groupForms, 'id'))
                ->where('disable',0)
                ->get();

            foreach ($groupForms as $formData) {
                $conflict = $existingForms->first(function ($form) use ($formData) {
                    return $form->order_of_cycle == $formData['order_of_cycle'] ||
                        $form->order_of_arrangement == $formData['order_of_arrangement'];
                });
                if ($conflict) {
                    $this->setError('Position conflict: Another form (ID: ' . $conflict->id .
                        ') already has the same cycle (' . $formData['order_of_cycle'] .
                        ') or arrangement (' . $formData['order_of_arrangement'] . ')');
                    return false;
                }
            }
        }
        return true;
    }


    private function performUpdates(array $formsData)
    {
        $updatedForms = [];
        foreach ($formsData as $data) {
            $form = Form::find($data['id']);
            $form->order_of_cycle = $data['order_of_cycle'];
            $form->order_of_arrangement = $data['order_of_arrangement'];
            if (isset($data['form_group_id'])) {
                $form->form_group_id = $data['form_group_id'];
            }
            $form->save();
            $updatedForms[] = $form;
        }
        return $updatedForms;
    }

    // End Of ARRANGE COMPLEX FORM //



    public function checkDisable($id)
    {
        $form = $this->formRepo->getParentForm($id);
        $versions = collect();
        $current = $form->parentForm;

        while ($current) {
            $versions->push([
                'id' => $current->id,
                'form_name' => $current->name,
                'date' => $current->disable_date,
            ]);
            $current = $current->parentForm;
        }
        return[
            'id' => $form->id,
            'form_name' => $form->name,
            'versions' => $versions->reverse()->values(),
        ];
    }
    public function getFormsVersions($id)
    {
        $form = $this->formRepo->getParentForm($id);
        $current = $form->parentForm;
        $versions[]=[
            'id'=>$form->id,
            'form_name'=>$form->name,
            'type'=>$form->type,
            "created_by"=>$form->creator->name ?? null,
            "updated_by"=>$form->updatedBy->name ?? null,
            'created_at'=>$form->created_at->format('Y-m-d H:i:s'),
            'updated_at'=>$form->updated_at->format('Y-m-d H:i:s')
        ];
        while ($current) {
            $versions[] =[
                'id' => $current->id,
                'form_name' => $current->name,
                'type'=>$form->type,
                "created_by" => $current->creator->name ?? null,
                "updated_by" => $current->updatedBy->name ?? null,
                'created_at' => $current->created_at->format('Y-m-d H:i:s'),
                'updated_at' => $current->updated_at->format('Y-m-d H:i:s')
            ];
            $current = $current->parentForm;
        }
        return $versions;
    }
    public function getGenralReport($data)
    {
        $rules = [
            'form_id' => 'required|exists:forms,id',
            'from' => "required|date|before_or_equal:to",
            'to' => "required|date|after_or_equal:from"
        ];
        $validator = validator($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());
            return false;
        }
        return $this->formRepo->getGeneralReports($data);
    }
    public function calclutorReports($data)
    {
        $rules = [
            'form_id' => 'required|exists:forms,id',
            'from' => "required|date|before_or_equal:to",
            'to' => "required|date|after_or_equal:from"
        ];
        $validator = validator($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());
            return false;
        }

        return $this->formRepo->getCalcetorReports($data);
    }
    public function getFormsHaveFiles($type = "general")
    {
        return $this->formRepo->getFormsHaveFiles($type);
    }
    public function getFileUrl($data)
    {
        if($forms = $this->formRepo->getFileReports($data))
            return $forms;

        $this->setError('Errors not found Files');
        return  false;
    }


}
