<?php

namespace App\Sys\Services\DepartmentForm;

use App\Models\Form;
use App\Sys\Repository\DepartmentForm\DepartmentFormRepo;
use App\Sys\Services;
use App\Sys\Services\StaticPage\StaticPageService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Exception;

class DepartmentFormService extends Services
{
    protected $deptFormRepo;

    public function __construct()
    {
        $this->deptFormRepo = new DepartmentFormRepo();
    }

    /**
     * Validate and attach a department to a form.
     *
     * Expected request body:
     * {
     *     "form_id": 1,
     *     "department_id": 3,
     *     "form_group_id": 5   // optional
     * }
     *
     * @param Request $request
     * @return Form
     *
     * @throws Exception
     */

    public function attachDepartment(Request $request)
    {
        $data = $request->all();
        $validator = Validator::make($data, [
            'form_id'       => 'required|exists:forms,id',
            "department_id"=>'required|array',
            'department_id.*' => 'exists:department,id',
            'form_group_id' => 'nullable|exists:form_groups,id',
        ]);

        if ($validator->fails()) {
            $this->setError($validator->errors());
            return false;
        }

        // Find the form using form_id from the request body.
        $form = Form::findOrFail($data['form_id']);

        $departmentId =$data['department_id'];
        $formGroupId = $data['form_group_id'] ?? null;

        // If form_group_id is provided, update the form record's column.
        if ($formGroupId) {
            $form->form_group_id = $formGroupId;
            $form->save();
        }

        // Attach the department using the repository
        // This will update the pivot table with form_group_id if provided.
        $this->deptFormRepo->attachDepartment($form, $departmentId, $formGroupId);

        // Return the updated form with its departments relation loaded.
        return $form->load('departments');
    }


    /**
     * Validate and detach a department from a form.
     *
     * Expected request body:
     * {
     *     "form_id": 1,
     *     "department_id": 3
     * }
     *
     * @param Request $request
     * @return Form
     *
     * @throws Exception
     */
    public function detachDepartment(Request $request)
    {
        $data = $request->all();
        $validator = Validator::make($data, [
            'form_id'       => 'required|exists:forms,id',
            'department_id' => 'required|exists:department,id',
        ]);

        if ($validator->fails()) {
            throw new Exception($validator->errors()->first());
        }

        // Find the form using form_id from the request body.
        $form = Form::findOrFail($data['form_id']);

        $departmentId = (int)$data['department_id'];
        $this->deptFormRepo->detachDepartment($form, $departmentId);

        // Return the updated form with its departments relation loaded.
        return $form->load('departments');
    }
}
