<?php

namespace App\Sys\Services\Auth;

use App\Models\User;
use App\Sys\Repository\Auth\AuthRepo;
use App\Sys\Services;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use validation;

class AuthService extends Services
{
    protected $authRepo;

    public function __construct()
    {
        $this->authRepo = new AuthRepo();
    }

    public function register($data)
    {
        // Ensure that only authenticated users can create a form.

        // This will use the currently authenticated user if available.
        if (!isset($data['user_id'])) {
            $data['user_id'] = auth()->id() ?? null;
        }


        if (!$this->validateData($data)) {
            return false;
        }

        $user = $this->authRepo->createUser($data);


        return [
            'user' => $user,
        ];
    }

    public function login(array $credentials)
    {
        $rules = [
            'login' => 'required|string|min:1', // This can be email or employee number
            'password' => 'required|string|min:6',
            'fcm_token' => 'string|nullable',
        ];

        $valid = validator($credentials, $rules);

        if ($valid->fails()) {
            $this->setError($valid->errors());
            return false;
        }

        $loginField = filter_var($credentials['login'] ?? '', FILTER_VALIDATE_EMAIL) ? 'email' : 'employee_number';

        $authCredentials = [
            $loginField => $credentials['login'],
            'password' => $credentials['password'],
        ];

        if (!$token = auth()->attempt($authCredentials)) {
            $this->setError('Invalid Credentials');
            return false;
        }

        // Get the authenticated user
        $user = Auth::user();

        // Update the fcm_token if provided
        if (!empty($credentials['fcm_token'])) {
            $user->update(['fcm_token' => $credentials['fcm_token']]);
        }

        // $refreshToken = auth()->guard('api')->refresh();

        $user->load('projects', 'branches','departments','role');


        return [
            'user' => $user,
            'token' => 'Bearer ' . $token,
             'expires_at' => now()->addMinutes(auth()->factory()->getTTL())->toDateTimeString(),
            //'refresh_token' => 'Bearer ' . $refreshToken,
            //            'projects' => $user->projects->map(function ($project) {
            //                return $project->only(['id', 'name']);
            //            }),
            //            'branches' => $user->branches->map(function ($branch) {
            //                return $branch->only(['id', 'name']);
            //            }),
            //            'departments' => [],
        ];
    }

    public function logout()
    {
        if (Auth::check()) {
            Auth::logout();
            return true;
        }
        return false;
    }

    public function getAllUsers() {}

    public function generateResetToken($email)
    {
        $rules = [
            'email' => 'required|email|exists:users,email',
        ];

        $validator = Validator::make(['email' => $email], $rules);
        if ($validator->fails()) {
            $this->setError('Invalid Email');
            return false;
        }

        $token = Str::random(40);

        // Store the token in the database
        $this->authRepo->storeToken($email, $token);

        return [
            'email' => $email,
            'token' => $token
        ];
    }


    public function resetPassword(array $data)
    {
        $rules = [
            'email' => 'required|email|exists:password_reset_tokens,email',
            'token' => 'required',
            'password' => 'required|string|min:6|confirmed',
        ];

        $validator = Validator::make($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());
        }

        $email = $data['email'];
        //        $token = $data['token'];
        $password = $data['password'];

        //        if (!$this->authRepo->isValidToken($email, $token)) {
        //            $this->setError('Invalid token');
        //            return false;
        //        }

        $this->authRepo->updatePassword($email, Hash::make($password));
        $this->authRepo->deleteToken($email);
        return true;
    }

    private function validateData(array $data): bool
    {
        $commonRules = [
            'name' => 'required|string|max:255',
            //'userType' => 'required|in:admin,branch,project',
            'role_id'=>'required|exists:roles,id',
            'email' => 'required|email|unique:users,email',
            'employee_number' => 'required|string|max:255|min:1|unique:users,employee_number',
            'password' => 'required|string|min:6|confirmed',
            'password_confirmation' => 'required|string|min:6',
        ];
        $validate = Validator::make($data, $commonRules);
        if ($validate->fails()) {
            $this->setError($validate->errors());
            return false;
        }
        return true;
    }

    private function getValidationRules(?string $userType)
    {

        $extraRules = [
            'branch' => [
                'phone' => 'nullable|string|max:15',
                //'fcm_token' => 'nullable|string'
            ],
            'project' => [
                // 'branch_id' => 'required|numeric',
                'phone' => 'nullable|string|max:15',
                //'hse_manager_name' => 'nullable|string|max:255',
                //'fcm_token' => 'nullable|string'
            ]
        ];

        return array_merge($commonRules, $extraRules[$userType] ?? []);
    }
}
