<?php

namespace App\Sys\Repository;

use App\Models\Branch;
use App\Models\Form;
use App\Models\FormColumn;
use App\Models\FormFiles;
use App\Models\FormInput;
use App\Models\FormInputOption;
use App\Models\FormRow;
use App\Models\FormSubColumn;
use App\Models\FormSubRow;
use App\Models\ProjectForm;
use App\Models\ProjectFormInput;
use App\Models\ProjectInput;
use App\Sys\ApiResponse;
use Illuminate\Support\Facades\DB;
use ZipArchive;
class FormRepo
{
    use ApiResponse;

    public function storeFrom($data)
    {
        $form = new Form();
        $form->name = $data['name'] ?? null;
        $form->form_group_id = $data['form_group_id'] ?? null;
        $form->description = $data['description'] ?? null;
        $form->type = $data['type'] ?? 'general';
        $form->step = $data['type'] == 'general' ? 'registereed' : 'compleated';
        $form->periodic_oneTime = $data['periodic_oneTime'] ?? 'oneTime';
        $form->periodic_type = $data['periodic_type'] ?? 'day';
        $form->deadline_by_num_of_days = $data['deadline_by_num_of_days'] ?? null;
        $form->week_day = $data['week_day'] ?? null;
        $form->month_day = $data['month_day'] ?? null;
        $form->quarter_start_day = $data['quarter_start_day'] ?? null;
        $form->approved = $data['approved'] ?? 0;
        $form->title_print = $data['title_print'] ?? null;
        $form->footer_print = $data['footer_print'] ?? null;
        $form->chart = $data['chart'] ?? 0;
        $form->is_active = $data['is_active'] ?? 1;
        //$form->order_of_cycle = $data['order_of_cycle'] ?? null;
        //$form->order_of_arrangement = $data['order_of_arrangement'] ?? null;
        $form->reference_id = $data['id'] ?? null;
        $form->onetime_send = $data['onetime_send'] ?? null;
        $form->user_id = auth()->id() ?? ($data['user_id'] ?? null);

        $form->save();

        // If the form is of type complex and belongs to a group,
        // and if the ordering values are missing, auto-assign them.
        if ($form->type == 'complex' && $form->form_group_id) {
            // Retrieve the maximum order_of_arrangement for existing complex forms in this group.
            $maxOrder = Form::where([
                ['form_group_id', $form->form_group_id],
                ['type', 'complex'],
                ['disable', 0]
            ])->count();

            // Default: all complex forms in the same group get cycle = 1,
            // and order_of_arrangement is maxOrder + 1 or 1 if none exists.
            $form->order_of_cycle = $maxOrder ?  $maxOrder + 1:1;
            $form->order_of_arrangement = $maxOrder ?  $maxOrder + 1:1;
            $form->save();

        }

        return $form;
    }

    public function getByBranch($id)
    {
        return Form::with('creator')->whereHas('getFormBranchProject', function ($q) use ($id) {
            $q->where('branch_id', $id);
        })->where('disable',0)->orderBy('created_at', 'desc')->get();
    }

    public function disable($id)
    {
        $form = Form::find($id);
        $form->disable = 1;
        $form->disable_date = date('Y-m-d');
        $form->updated_by = Auth()->user()->id;
        return $form->save();
    }

    public function updateFormNext($form_id, $nextDate)
    {
        $from = Form::find($form_id);
        $from->next_send_date = $nextDate;
        return $from->save();
    }

    public function storeRow($form, $data)
    {
        $row = new FormRow();
        $row->form_id = $form->id;
        $row->single_multi_column = $data['single_multi_column'] ?? 'single_column';
        $row->single_multi_input = $data['single_multi_input'] ?? 'single_input';
        $row->type_row = $data['type_row'] ?? 'td';
        $row->save();

        return $row;
    }

    public function storeColumn($row, $data)
    {
        $column = new FormColumn();
        $column->form_row_id = $row->id;
        $column->width = $data['width'] ?? 100;
        $column->column_type = $data['column_type'] ?? 'input';
        $column->save();
        return $column;
    }

    public function storeInput($form, $column, $data, $row, $sub_row = null, $subColumn = null)
    {
        $input = new FormInput();
        $input->form_column_id = $column->id;
        $input->form_id = $form->id;
        $input->form_row_id = $row->id;
        $input->form_sub_column_id = $subColumn;
        $input->form_sub_row_id = $sub_row;
        $input->label = $data['label'] ?? null;
        $input->input_type = $data['input_type'] ?? 'text';
        $input->placeholder = $data['placeholder'] ?? null;
        $value = $data['value'] ?? null;
        $input->value = is_array($value) ? json_encode($value) : $value;
        $input->calculated = $data['calculated'] ?? 0;
        $input->required = $data['required'] ?? 0;
        $input->chart = $data['chart'] ?? 0;
        $input->show_pdf = $data['show_pdf'] ?? 1;
        $input->show_on_fill = $data['show_on_fill'] ?? 1;
        $input->chart_type = $data['chart_type'] ?? 'fixed';
        $input->icon = $data['icon'] ?? null;
        $input->writing_mode = $data['writing_mode'] ?? "vertical-rl";
        $input->save();

        return $input;
    }

    public function storeInputOption($input, $data)
    {
        $option = new FormInputOption();
        $option->input_id = $input->id;
        $option->name = $data['name'] ?? null;
        $option->save();

        return $option;
    }

    public function FromByNextDate($next)
    {
        return Form::with('form_input')->where([
            ['type', 'general'],
            ['is_active', 1],
            ['next_send_date', $next]
        ])->get();
    }

    public function getFormById($id)
    {
        return Form::with('form_input','creator')->find($id);
    }

    public function getParentForm($id)
    {
        return Form::with('parentForm','creator','updatedBy')->find($id);
    }

    public function updatedStep($id)
    {
        $form = Form::find($id);
        $form->step = "compleated";
        return $form->save();
    }

    public function getFormByType()
    {
        $name = request()->name;
        $type = request()->type ?? 'general';
        $limit = request()->limit;
        return Form::with('creator')->where('type', $type)
            ->when($name, function ($q) use ($name) {
                $q->where('name', 'like', '%' . $name . '%');
            })->where('disable',0)->orderBy('created_at', 'desc')->paginate($limit);
    }

    public function updateForm($id, $data)
    {
        $form = Form::find($id);

        if (!$form) {
            return false;
        }

        $form->name = $data['name'] ?? $form->name;
        $form->form_group_id = $data['form_group_id'] ?? $form->form_group_id;
        $form->description = $data['description'] ?? $form->description;
        $form->chart = $data['chart'] ?? $form->chart;
        $form->title_print = $data['title_print'] ?? $form->title_print;
        $form->footer_print = $data['footer_print'] ?? $form->footer_print;
        $form->order_of_cycle = $data['order_of_cycle'] ?? null;
        $form->order_of_arrangement = $data['order_of_arrangement'] ?? null;
        $form->is_active = $data['is_active'] ?? $form->is_active;
        $form->approved = $data['approved'] ?? $form->approved;
        $form->user_id = auth()->id() ?? ($data['user_id'] ?? null);

        $form->save();

        return $form;
    }

    public function updateFormInput($inputId, $data)
    {
        $input = FormInput::find($inputId);

        if (!$input) {
            return false;
        }

        $input->label = $data['label'] ?? $input->label;
        $input->placeholder = $data['placeholder'] ?? $input->placeholder;
        $input->chart_type = $data['chart_type'] ?? $input->chart_type;
        $input->required = $data['required'] ?? $input->required;
        $input->show_pdf = $data['show_pdf'] ?? $input->show_pdf;

        $input->save();

        return $input;
    }

    public function getFullFormById($formId)
    {
        return Form::with('rows.columns.inputs.options')->findOrFail($formId);
    }

    public function addFormFile($form_id, $name, $type, $file_name)
    {
        $form = new FormFiles();
        $form->form_id = $form_id;
        $form->type = $type;
        $form->path = $name;
        $form->file_name = $file_name;
        return $form->save();
    }

    public function getFullFormWithAnswersByProjectFormId($projectFormId)
    {
        $projectForm = ProjectForm::with([
            'form.rows.columns.inputs.options', // Ensure options are loaded for inputs
            'form.rows.columns.getsubRows.getSubColumn.getInput.getOption'
        ])->findOrFail($projectFormId);

        $projectInputs = ProjectInput::where('project_form_id', $projectFormId)
            ->with('inputs.options') // Ensure options are loaded for project inputs
            ->get()
            ->keyBy('form_input_id');

        // dd($projectInputs);
        return [
            'form' => $projectForm->form,
            'projectInputs' => $projectInputs,
            'reference' => $projectForm->reference_code
        ];
    }

    public function getAll($types)
    {
        return Form::with('formGroup','creator')->where('disable', 0)->whereIn('type', $types)->orderBy('created_at', 'desc')->get();
    }

    public function storeSubRow($column, $data)
    {
        $sub = new FormSubRow();
        $sub->column_id = $column->id;
        $sub->single_multi_column = $data['single_multi_column'] ?? 'single_column';
        $sub->single_multi_input = $data['single_multi_input'] ?? 'single_input';
        $sub->type_row = $data['type_row'] ?? 'td';
        $sub->save();
        return $sub;
    }

    public function storeSubCloum($column, $row, $data)
    {
        $sub = new FormSubColumn();
        $sub->form_sub_row_id = $row->id;
        $sub->column_id = $column->id;
        $sub->width = $data['width'] ?? 100;
        $sub->save();
        return $sub;
    }

    public function getbyId($id)
    {
        return Form::with(['creator','formGroup', 'getRow', 'files', 'ProjectForms' => function ($query) {
            $query->latest('created_at')->limit(1);
        }])->find($id);
    }

    public function compalteChart()
    {
        return Form::select('forms.name as form_name')
            ->selectRaw("COUNT(*) as complete_count")
            ->join('project_forms', 'forms.id', '=', 'project_forms.form_id')
            ->whereIn('project_forms.status', ['expired', 'done', 'approved'])
            ->where([
                ['project_forms.chart', 1],
                ['project_forms.type', 'general']
            ])
            ->groupBy('forms.name')
            ->get();
    }

    public function chartCompalteInCompBranch()
    {
        return Branch::select('branches.name')
            ->selectRaw("SUM(CASE WHEN project_forms.status IN ('done','expired' ,'approved') THEN 1 ELSE 0 END) as complete_forms")
            ->selectRaw("SUM(CASE WHEN project_forms.status IN ('waiting','waiting_approved') THEN 1 ELSE 0 END) as not_complete_forms")
            ->join('project_forms', 'branches.id', '=', 'project_forms.branch_id')
            ->where([
                ['project_forms.chart', 1],
                ['project_forms.type', 'general']
            ])
            ->groupBy('branches.name')
            ->get();
    }

    public function count()
    {
        return Form::count();
    }

    public function getFromChartFixed($type)
    {
        return Form::whereHas('form_input', function ($query) {
            $query->where('input_type', 'number')
                ->where('chart', 1)
                ->where('chart_type', 'fixed');
        })->where([
            ['type', $type],
            ['disable', 0]
        ])->get();
    }

    public function getFormsChartChart($type,$branch)
    {
        return Form::whereHas('ProjectForms',function ($query) use ($branch,$type){
            $query->where('branch_id', $branch)
                ->where('type', $type);
        })->whereHas('form_input', function ($query) {
            $query->where('input_type', 'number')
                ->where('chart', 1)
                ->where('chart_type', 'chart');
        })->where([
            ['type', $type],
            ['disable', 0]
        ])->get();
    }

    public function getFormChartInput($id)
    {
        return FormInput::where([
            ['form_id', '=', $id],
            ['input_type', '=', 'number'],
            ['chart', '=', 1],
            ['chart_type', '=', 'chart'],
        ])->get();
    }

    public function getFormsBygroup($group, $id)
    {
        return Form::where([
            ['form_group_id', $group],
            ['disable', 0]
        ])->whereNotIn('id', [$id])->get();
    }

    public function getFormsgroup($group_id)
    {
        return Form::where([
            ['form_group_id', $group_id],
            ['disable', 0]
        ])->get();
    }

    public function getGeneralReports($data)
    {
        $form = Form::where('id', $data['form_id'])->first();
        $startDate = date('Y-m-d', strtotime($data['from']));
        $endDate = date('Y-m-d', strtotime($data['to'] . ' +1 day'));
        $project = $data['project_id'] == 0 ? null : $data['project_id'];
        $branch = $data['branch_id'] == 0 ? null : $data['branch_id'];
        $project_forms = ProjectForm::with(['project', 'projectFormInputs.inputs'])
            ->when($branch, function ($q) use ($branch) {
                $q->whereIn('branch_id', $branch);
            })->when($project, function ($q) use ($project) {
                $q->whereIn('project_id', $project);
            })->whereBetween('created_at', [$startDate, $endDate])
            ->where('form_id', $form->id)
            ->get();

        $headers = [];
        $body = [];
        $inputArrIds = [];

        foreach ($project_forms as $pform) {
            $originalHeader = $pform->project->name . '-_' . $pform->id;
            $cleanHeader = preg_replace('/-_\d+$/', '', $originalHeader);
            $headers[] = $cleanHeader;
        }

        foreach ($project_forms as $pform) {
            foreach ($pform->projectFormInputs as $input) {
                $formInput = $input->inputs;
                if ($formInput->input_type != "label" && !in_array($formInput->id, $inputArrIds)) {
                    $row = [
                        'id' => $formInput->id,
                        'label' => $formInput->label,
                        'answers' => [],
                        'sum' => 0,
                    ];
                    $calculatedSum = 0;
                    foreach ($project_forms as $pform2) {
                        $answer_value = $pform2->projectFormInputs->where('form_input_id', $formInput->id)->first();
                        $tdVal = '';

                        if (!$answer_value) {
                            $row['answers'][] = '';
                            continue;
                        }

                        switch ($answer_value->input_type) {
                            case 'file':
                                if ($answer_value->file_type == 'application/pdf') {
                                    $tdVal = $answer_value->answer ? url($answer_value->answer) : '';
                                } else {
                                    $tdVal = $answer_value->answer ? url($answer_value->answer) : '';
                                }
                                break;

                            case 'checkbox':
                                $answer = $answer_value->answer ? json_decode($answer_value->answer, true) : [0];
                                $checkboxNames = FormInputOption::whereIn('id', $answer)->pluck('name')->toArray();
                                $tdVal = !empty($checkboxNames) ? '( ' . implode(' , ', $checkboxNames) . ' )' : '';
                                break;

                            case 'radio':
                                $element = FormInputOption::find($answer_value->answer);
                                $tdVal = $element ? '( ' . $element->name . ' )' : '';
                                break;

                            default:
                                $tdVal = $answer_value->answer;
                                break;
                        }

                        if ($answer_value->calculated == 1) {
                            $calculatedSum += is_numeric($answer_value->answer) ? $answer_value->answer : 0;
                        } else {
                            $calculatedSum = 'N/A';
                        }

                        $row['answers'][] = $tdVal;
                    }
                    $row['sum'] = $calculatedSum;
                    $body[] = $row;
                    $inputArrIds[] = $formInput->id;
                }
            }
        }

        return [
            'headers' => $headers,
            'body' => $body
        ];

    }

    public function getCalcetorReports($data)
    {
        $startDate = date('Y-m-d', strtotime($data['from']));
        $endDate = date('Y-m-d', strtotime($data['to'] . ' +1 day'));
        $project = $data['project_id'] == 0 ? null : $data['project_id'];
        $branch = $data['branch_id'] == 0 ? null : $data['branch_id'];
        $project_forms = ProjectForm::where('form_id', $data['form_id'])
            ->when($branch, function ($q) use ($branch) {
                $q->whereIn('branch_id', $branch);
            })->when($project, function ($q) use ($project) {
                $q->whereIn('project_id', $project);
            })->pluck('id');

        return $inputs = ProjectInput::with('inputs')
            ->where(['input_type' => 'number', 'calculated' => 1])
            ->whereIn('project_form_id', $project_forms)
            ->whereBetween('created_at', [$startDate, $endDate])
            ->select('form_input_id', DB::raw('SUM(answer) as total'))
            ->groupBy('form_input_id')
            ->get();

    }

    public function getFormsHaveFiles($type)
    {
        return Form::whereHas('form_input', function ($query) {
            $query->where('input_type', 'file');
        })->where([
            ['type', $type],
            ['disable', 0]
        ])->get();
    }

    public function getFileReports($data)
    {
        $startDate = date('Y-m-d', strtotime($data['from']));
        $endDate = date('Y-m-d', strtotime($data['to'] . ' +1 day'));
        $project = $data['project_id'] == 0 ? null : $data['project_id'];
        $branch = $data['branch_id'] == 0 ? null : $data['branch_id'];
        $forms = $data['form_id'];
        $form = Form::find($data['form_id']);
        $project_forms = ProjectForm::whereBetween('created_at', [$startDate, $endDate])
            ->when($branch, function ($q) use ($branch) {
                $q->whereIn('branch_id', $branch);
            })->when($project, function ($q) use ($project) {
                $q->whereIn('project_id', $project);
            })->where('form_id', $forms)
            ->get();

        $zipFileName = "$form->name.zip";
        $zip = new ZipArchive;
        $filesToZip = [];

        foreach ($project_forms as $pform) {
            $project_name = $pform->branch->name . '/' . $pform->project->name;
            $inputs = ProjectInput::with('inputs')
                ->where('project_form_id', $pform->id)
                ->where('input_type', 'file')
                ->get();

            foreach ($inputs as $input) {
                $path = public_path($input->answer);
                if ($input->answer != null && file_exists($path)) {
                    $fileExtension = pathinfo($path, PATHINFO_EXTENSION);
                    $fileName = $input->inputs->label . '--' . $input->id . '--' . $input->updated_at->format('Y-m-d') . '.' . $fileExtension;
                    $filesToZip[$project_name][$fileName] = $path;
                }
            }
        }

        if (!empty($filesToZip)) {
            $zipFileName = $form->name . '_' . time() . '.zip';
            $zipDirectory = public_path('zips');

            if (!file_exists($zipDirectory)) {
                mkdir($zipDirectory, 0777, true);
            }

            $zipFilePath = $zipDirectory . '/' . $zipFileName;

            if ($zip->open($zipFilePath, ZipArchive::CREATE) === TRUE) {
                foreach ($filesToZip as $project => $files) {
                    foreach ($files as $name => $absoluteFilePath) {
                        if (file_exists($absoluteFilePath)) {
                            $zip->addFile($absoluteFilePath, "$project/$name");
                        }
                    }
                }
                $zip->close();


                $url = url("zips/" . $zipFileName);
                return $url;
            }
        }
        return false;
    }
}
