<?php

namespace App\Http\Controllers\FormGroupArrange;

use App\Http\Controllers\Controller;
use App\Http\Resources\FormGroupArrange\FormGroupArrangeResource;
use App\Sys\ApiResponse;
use App\Sys\Services\FormGroupArrange\FormGroupArrangeService;
use Illuminate\Http\Request;

class FormGroupArrangeController extends Controller
{
    use ApiResponse;

    private $arrangeService;

    public function __construct()
    {
        $this->arrangeService = new FormGroupArrangeService();
    }

    /**
     * Get arrangements by form group id.
     */
    public function getByGroup($groupId)
    {
        $arrangements = $this->arrangeService->getByGroup($groupId);
        if (!$arrangements) {
            return $this->apiResponse(400, 'Error', 'No arrangements found for this group', null);
        }
        return $this->apiResponse(200, 'Arrangements retrieved successfully', null, FormGroupArrangeResource::collection($arrangements));
    }

    /**
     * Sync all arrangements for a form group (create/replace)
     */
    public function syncArrangements(Request $request, $formGroupId)
    {
        $result = $this->arrangeService->syncArrangements($formGroupId, $request->all());

        if ($result) {
            return $this->apiResponse(
                200,
                'Arrangements synced successfully',
                null,
                FormGroupArrangeResource::collection($result)
            );
        }

        return $this->apiResponse(400, 'Error', $this->arrangeService->errors(), null);
    }

    /**
     * Update multiple arrangements for a form group
     */
    public function updateArrangements(Request $request, $formGroupId)
    {
        $result = $this->arrangeService->updateArrangements($formGroupId, $request->all());

        if ($result) {
            return $this->apiResponse(
                200,
                'Arrangements updated successfully',
                null,
                FormGroupArrangeResource::collection($result)
            );
        }

        return $this->apiResponse(400, 'Error', $this->arrangeService->errors(), null);
    }

    /**
     * Display a listing of the arrangements.
     */
    public function index()
    {
        $arrangements = $this->arrangeService->get();
        return $this->apiResponse(200, 'All Form Group Arrangements', null, FormGroupArrangeResource::collection($arrangements));
    }

    /**
     * Store a newly created arrangement.
     */
    public function store(Request $request)
    {
        $result = $this->arrangeService->add($request->all());
        if ($result) {
            return $this->apiResponse(200, 'Arrangement created successfully', null, new FormGroupArrangeResource($result));
        }
        return $this->apiResponse(400, 'Error', $this->arrangeService->errors(), null);
    }

    /**
     * Display the specified arrangement.
     */
    public function show($id)
    {
        $record = $this->arrangeService->getById($id);
        if (!$record) {
            return $this->apiResponse(400, 'Error', 'Record not found', null);
        }
        return $this->apiResponse(200, 'Success', null, new FormGroupArrangeResource($record));
    }

    /**
     * Update the specified arrangement.
     */
    public function update(Request $request, $id)
    {
        $data = $request->all();
        $data['id'] = $id;
        $result = $this->arrangeService->update($data);
        if ($result) {
            return $this->apiResponse(200, 'Arrangement updated successfully', null, new FormGroupArrangeResource($result));
        }
        return $this->apiResponse(400, 'Error', $this->arrangeService->errors(), null);
    }

    /**
     * Remove the specified arrangement.
     */
    public function destroy($id)
    {
        $result = $this->arrangeService->del($id);
        if ($result) {
            return $this->apiResponse(200, 'Arrangement deleted successfully', null, null);
        }
        return $this->apiResponse(400, 'Error', $this->arrangeService->errors(), null);
    }
}
