<?php

namespace App\Sys\Services\StaticPage;

use App\Sys\Repository\StaticPage\StaticPageRepository;
use App\Sys\Services;
use App\Sys\Services\Base64FileUploader;

class StaticPageService extends Services
{
    private $staticPage;

    public function __construct()
    {
        $this->staticPage = new StaticPageRepository();
    }

    public function add($data)
    {
        if (!isset($data['user_id'])) {
            $data['user_id'] = auth()->id() ?? null;
        }

        $rules = [
            'title_en' => 'required|string|max:255',
            'title_ar' => 'required|string|max:255',
            'body_en'  => 'required|string',
            'body_ar'  => 'required|string',
            // Cover is optional and should be a Base64 string if provided.
            'cover'    => 'required|string'
        ];
        $valid = validator($data, $rules);
        if ($valid->fails()) {
            $this->setError($valid->errors());
            return false;
        }

        // Process the cover image if provided.
        if (!empty($data['cover'])) {
            $storedPath = Base64FileUploader::saveBase64File($data['cover'], 'staticPage');
            if ($storedPath) {
                $data['cover'] = $storedPath;
            } else {
                $this->setError('Invalid base64 file data');
                return false;
            }
        }

        return $this->staticPage->create($data);
    }

    public function updated($data)
    {
        // Ensure that only authenticated users can create a form.
        // This will use the currently authenticated user if available.
        $rules = [
            'id'       => 'required',
            'title_en' => 'sometimes|string|max:255',
            'title_ar' => 'sometimes|string|max:255',
            'body_en'  => 'sometimes|string',
            'body_ar'  => 'sometimes|string',
            'cover'    => 'nullable|string'
        ];
        $valid = validator($data, $rules);
        if ($valid->fails()) {
            $this->setError($valid->errors());
            return false;
        }
        // If a new cover image is provided, remove the old file and process the new one.
        if (!empty($data['cover'])) {
            $storedPath = Base64FileUploader::saveBase64File($data['cover'], 'staticPage');
            if ($storedPath) {
                $data['image'] = $storedPath;
            } else {
                $this->setError('Invalid base64 file data');
                return false;
            }
        }

        return $this->staticPage->update($data['id'], $data);
    }

    public function get()
    {
        return $this->staticPage->all();
    }

    public function getByid($id)
    {
        return $this->staticPage->find($id);
    }

    public function del($id)
    {
        $page = $this->staticPage->find($id);
        if (!empty($page)) {
            if ($page->cover && file_exists(public_path($page->cover))) {
                unlink(public_path($page->cover));
            }
            return $this->staticPage->delete($id);
        }
        $this->setError('Not Found');
        return false;
    }
}
