<?php

namespace App\Sys\Services;

use Illuminate\Support\Str;
use Illuminate\Support\Facades\File;

class Base64FileUploader
{
    /**
     * Save a Base64 encoded image file into the public/uploads/staticPage folder.
     *
     * @param string $base64String The Base64 string including data header.
     * @param string $folderName   The folder name under public/uploads (default: staticPage).
     * @return string|null         Relative path of the saved file or null on failure.
     */
    public static function saveBase64File(string $base64String, string $folderName = 'staticPage'): ?string
    {
        // Validate that the string contains a Base64 header (example: data:image/png;base64,...)
        if (!preg_match('/^data:(?<mime>image\/\w+);base64,/', $base64String, $matches)) {
            return null;
        }

        $mimeType = $matches['mime'];

        // Map MIME types to file extensions.
        $mimeExtensions = [
            'image/jpeg' => 'jpg',
            'image/jpg'  => 'jpg',
            'image/png'  => 'png',
            'image/gif'  => 'gif',
            'application/pdf' => 'pdf',
        ];
        $extension = $mimeExtensions[$mimeType] ?? 'png';

        // Remove the header and decode the Base64 data.
        $base64Data = preg_replace('/^data:image\/\w+;base64,/', '', $base64String);
        $decodedData = base64_decode($base64Data);
        if ($decodedData === false) {
            return null;
        }

        // Generate a unique filename.
        $filename = Str::random(40) . '.' . $extension;
        $uploadDir = public_path('uploads/' . $folderName);

        // Ensure the upload directory exists.
        if (!File::exists($uploadDir)) {
            File::makeDirectory($uploadDir, 0777, true, true);
        }

        $fullPath = $uploadDir . '/' . $filename;

        // Save the file data.
        if (file_put_contents($fullPath, $decodedData) === false) {
            return null;
        }

        // Return the relative path that can be stored in the database.
        return 'uploads/' . $folderName . '/' . $filename;
    }
}
