<?php

namespace App\Sys\Repository\User;

use App\Models\Role;
use App\Models\User;
use App\Models\UserBranch;
use App\Models\UserDepartment;
use App\Models\UserProject;
use App\Sys\LogsModelActivity;

class UserRepo
{
    use LogsModelActivity;
    public function findById($userId)
    {
        return User::with('branches', 'projects', 'departments', 'creator', 'role')->find($userId);
    }


    public function getByid($userId)
    {
        return User::with('branches', 'projects', 'departments', 'creator', 'role')->find($userId);
    }

    public function existsByEmail($email, $userId = null)
    {
        $query = User::where('email', $email);
        if ($userId) {
            $query->where('id', '!=', $userId);
        }
        return $query->exists();
    }

    public function update($userId, $data)
    {
        $user = $this->findById($userId);
        if (!$user) {
            return false;
        }
        $oldRole = Role::find($user->role_id);
        $originalBranch = clone $user;
        $user->load('branches', 'projects', 'departments');
        $oldRelations = [
            'branches' => $user->branches->map(fn($b) => ['id' => $b->id, 'name' => $b->name])->toArray(),
            'projects' => $user->projects->map(fn($p) => ['id' => $p->id, 'name' => $p->name])->toArray(),
            'departments' => $user->departments->map(fn($d) => ['id' => $d->id, 'name' => $d->name])->toArray(),
        ];
        $user->name = $data['name'];
        $user->email = $data['email'];
        $user->employee_number = $data['employee_number'];
        $user->role_id = $data['role_id'];
        $user->phone = $data['phone'] ?? null;
        $user->user_id = auth()->check() ? auth()->user()->id : null;
        $user->save();

        if (!empty($data['branches']) && is_array($data['branches'])) {
            UserBranch::where('user_id', $user->id)->delete();
            foreach ($data['branches'] as $b) {
                $userBranch = new UserBranch();
                $userBranch->user_id = $user->id;
                $userBranch->branch_id = $b;
                $userBranch->save();
            }
        }

        if (!empty($data['project']) && is_array($data['project'])) {
            UserProject::where('user_id', $user->id)->delete();
            foreach ($data['project'] as $b) {
                $userBranch = new UserProject();
                $userBranch->user_id = $user->id;
                $userBranch->project_id = $b;
                $userBranch->save();
            }
        }
        // Attach departments via pivot table if provided

        if (!empty($data['departments'])) {
            UserDepartment::where('user_id', $user->id)->delete();
            foreach ($data['departments'] as $d) {
                // Expecting $data['departments'] to be an array of department IDs.
                $userDepartment = new UserDepartment();
                $userDepartment->user_id = $user->id;
                $userDepartment->department_id = $d;
                $userDepartment->save();
            }
        }

        $user->load('branches', 'projects', 'departments');
        $newRelations = [
            'branches'    => $user->branches->map(fn($b) => ['id' => $b->id, 'name' => $b->name])->toArray(),
            'projects'    => $user->projects->map(fn($p) => ['id' => $p->id, 'name' => $p->name])->toArray(),
            'departments' => $user->departments->map(fn($d) => ['id' => $d->id, 'name' => $d->name])->toArray(),
        ];
        $role = Role::find($user->role_id);
        $realtion =[
            'role'=>['old' => collect([$oldRole]),'new' =>$role ? collect([$role]) : collect()],
            'branches'=>['old'=>$oldRelations['branches'],'new'=>$newRelations['branches']],
            'projects'=>['old'=>$oldRelations['projects'],'new'=>$newRelations['projects']],
            'departments'=>['old'=>$oldRelations['departments'],'new'=>$newRelations['departments']],
        ];
        $this->logActivity('user',$user,'updated',$realtion,$originalBranch);
        return $user;
    }

    public function getAllUsers($perPage)
    {
        return User::paginate($perPage);
    }

    public function getUserByType($type = "project")
    {
        return User::where('userType', $type)->get();
    }

    public function searchUsers()
    {
        $name = request()->name;
        $type = request()->type;
        $limit = request()->perPage;
        $email = request()->email;
        $phone = request()->phone;
        $branch_name = request()->branch_name;
        $project_name = request()->project_name;
        $branch = request()->has('branch') && request()->branch !== null ? explode(',', request()->branch) : null;
        $project = request()->has('project') && request()->project !== null ? explode(',', request()->project) : null;
        $department = request()->has('department') && request()->department !== null ? explode(',', request()->department) : null;

        return User::with('branches', 'projects', 'departments', 'creator', 'role')
            ->when($type, function ($q) use ($type) {
                $q->where('userType', $type);
            })->when($name, function ($q) use ($name) {
                $q->where('name', 'like', '%' . $name . '%');
            })->when($email, function ($q) use ($email) {
                $q->where('name', 'like', '%' . $email . '%');
            })->when($phone, function ($q) use ($phone) {
                $q->where('phone', 'like', '%' . $phone . '%');
            })->when($name, function ($q, $name) {
                $q->whereHas('form', function ($q) use ($name) {
                    $q->where('name', 'like', '%' . $name . '%');
                });
            })->when($branch, function ($q, $branch) {
                $q->whereHas('userBranches', function ($q) use ($branch) {
                    $q->whereIn('branch_id', $branch);
                });
            })->when($branch_name, function ($q, $branch_name) {
                $q->whereHas('userBranches.branch', function ($q) use ($branch_name) {
                    $q->where('name', 'like', '%' . $branch_name . '%');
                });
            })->when($project, function ($q, $project) {
                $q->whereHas('userProjects', function ($q) use ($project) {
                    $q->whereIn('project_id', $project);
                });
            })->when($project_name, function ($q, $project_name) {
                $q->whereHas('userProjects.project', function ($q) use ($project_name) {
                    $q->where('name', 'like', '%' . $project_name . '%');
                });
            })->when($department, function ($q, $department) {
                $q->whereHas('departments', function ($q) use ($department) {
                    $q->whereIn('department_id', $department);
                });
            })->paginate($limit);
    }
}
