<?php

namespace App\Http\Controllers\NotificationBar;

use App\Http\Controllers\Controller;
use App\Http\Resources\NotificationBarResource;
use App\Http\Resources\NotificationBarCollection;
use App\Sys\ApiResponse;
use App\Sys\Services\NotificationBar\NotificationBarServices;
use Illuminate\Http\Request;
use Exception;

class NotificationBarController extends Controller
{
    use ApiResponse;

    private $notifService;

    public function __construct()
    {
        // Instantiate the service layer.
        $this->notifService = new NotificationBarServices();
    }

    /**
     * Retrieve all notification bars.
     */
    public function index()
    {
        try {
            $data = $this->notifService->get();
            $collection = new NotificationBarCollection($data);
            return $this->apiResponse(200, 'Notification Bars retrieved successfully', null, $collection);
        } catch (Exception $e) {
            return $this->apiResponse(500, 'Error retrieving notification bars', [$e->getMessage()], null);
        }
    }

    /**
     * Create a new notification bar.
     */
    public function store(Request $request)
    {
        try {
            $data = $request->all();
            $result = $this->notifService->add($data);
            if ($result === false) {
                $errors = $this->notifService->errors();
                return $this->apiResponse(400, 'Error creating notification bar', $errors, null);
            }
            $resource = new NotificationBarResource($result);
            return $this->apiResponse(200, 'Notification Bar created successfully', null, $resource);
        } catch (Exception $e) {
            return $this->apiResponse(400, 'Error creating notification bar', [$e->getMessage()], null);
        }
    }

    /**
     * Retrieve a single notification bar by ID.
     */
    public function show($id)
    {
        try {
            $data = $this->notifService->getByid($id);
            $resource = new NotificationBarResource($data);
            return $this->apiResponse(200, 'Notification Bar retrieved successfully', null, $resource);
        } catch (Exception $e) {
            return $this->apiResponse(404, 'Notification Bar not found', [$e->getMessage()], null);
        }
    }

    /**
     * Update an existing notification bar.
     */
    public function update(Request $request, $id)
    {
        try {
            $data = $request->all();
            $data['id'] = $id;
            $result = $this->notifService->updated($data);
            if ($result === false) {
                $errors = $this->notifService->errors();
                return $this->apiResponse(400, 'Error updating notification bar', $errors, null);
            }
            $resource = new NotificationBarResource($result);
            return $this->apiResponse(200, 'Notification Bar updated successfully', null, $resource);
        } catch (Exception $e) {
            return $this->apiResponse(400, 'Error updating notification bar', [$e->getMessage()], null);
        }
    }

    /**
     * Delete a notification bar.
     */
    public function destroy($id)
    {
        try {
            $result = $this->notifService->del($id);
            if ($result === false) {
                $errors = $this->notifService->errors();
                return $this->apiResponse(400, 'Error deleting notification bar', $errors, null);
            }
            return $this->apiResponse(200, 'Notification Bar deleted successfully', null, null);
        } catch (Exception $e) {
            return $this->apiResponse(400, 'Error deleting notification bar', [$e->getMessage()], null);
        }
    }

    /**
     * Retrieve only active notification bars.
     */
    public function getActive()
    {
        try {
            $data = $this->notifService->getActive();
            $collection = new NotificationBarCollection($data);
            return $this->apiResponse(200, 'Active Notification Bars retrieved successfully', null, $collection);
        } catch (Exception $e) {
            return $this->apiResponse(500, 'Error retrieving active notification bars', [$e->getMessage()], null);
        }
    }
}
