<?php

namespace App\Http\Controllers\FormGroup;

use App\Http\Controllers\Controller;
use App\Http\Resources\FormGroup\FormGroupCollection;
use App\Http\Resources\FormGroup\FormGroupResource;
use App\Http\Resources\FormsCollection;
use App\Http\Resources\FormsResource;
use App\Sys\ApiResponse;
use App\Sys\Services\FormGroup\FormGroupService;
use Illuminate\Http\Request;

class FormGroupController extends Controller
{
    use ApiResponse;

    private $formGroupService;

    public function __construct()
    {
        $this->formGroupService = new FormGroupService();
    }

    /**
     * Display a listing of form groups.
     */
    public function index()
    {
        $groups = $this->formGroupService->get();
        $collection = new FormGroupCollection($groups);
        return $this->apiResponse(200, 'All Form Groups', null, $collection);
    }

    /**
     * Create a new form group.
     */

    public function store(Request $request)
    {
        $result = $this->formGroupService->add($request->all());
        if ($result) {
            return $this->apiResponse(200, 'Form group created successfully', null, new FormGroupResource($result));
        }
        return $this->apiResponse(400, 'Error', $this->formGroupService->errors(), null);
    }

    /**
     * Display a specific form group.
     */
    public function show($id)
    {
        $group = $this->formGroupService->getByid($id);
        if (!$group) {
            return $this->apiResponse(400, 'Error', 'Form group not found', null);
        }
        return $this->apiResponse(200, 'Success', null, new FormGroupResource($group));
    }

    /**
     * Update a specific form group.
     */

    public function update(Request $request, $id)
    {
        $data = $request->all();
        $data['id'] = $id;
        $result = $this->formGroupService->update($data);
        if ($result) {
            return $this->apiResponse(200, 'Form group updated successfully', null, new FormGroupResource($result));
        }
        return $this->apiResponse(400, 'Error', $this->formGroupService->errors(), null);
    }

    /**
     * Delete a specific form group.
     */
    public function destroy($id)
    {
        $result = $this->formGroupService->del($id);
        if ($result) {
            return $this->apiResponse(200, 'Form group deleted successfully', null, null);
        }
        return $this->apiResponse(400, 'Error', $this->formGroupService->errors(), null);
    }

    /**
     * Get forms filtered by department and/or form name with pagination.
     *
     * Query parameters:
     * - department: Department id or name (optional)
     * - form_name: Form name to search for (optional)
     * - limit: Number of results per page (optional, default is 10)
     *
     * Example:
     * GET /api/form-groups/by-filters?department=Sales&form_name=Invoice&limit=15
     *
     * @param Request $request
     * @return \Illuminate\Http\Response
     */
    // public function getFormsByDepartment(Request $request)
    // {
    //     $department = $request->query('department');
    //     $formName = $request->query('form_name');
    //     $limit = $request->query('limit', 10);

    //     // If neither filter is provided, you might choose to return an error
    //     if (!$department && !$formName) {
    //         return $this->apiResponse(400, 'Error', 'At least one filter (department or form_name) is required', null);
    //     }

    //     $forms = $this->formGroupService->getFormsByFilters($department, $formName, $limit);

    //     // Returning a resource collection with pagination details
    //     return $this->apiResponse(200, 'Forms retrieved successfully', null, new FormsCollection($forms));
    // }

    public function getFormsByDepartment(Request $request)
    {
        // Get parameters
        $departmentIds = $request->query('department_ids');
        $formName = $request->query('form_name');
        $limit = $request->query('limit', 10);

        // Convert department_ids to array if it's not null
        if ($departmentIds && !is_array($departmentIds)) {
            $departmentIds = explode(',', $departmentIds);
        }

        // Validate at least one filter is provided
        if (!$departmentIds && !$formName) {
            return $this->apiResponse(400, 'Error', 'At least one filter (department_ids or form_name) is required', null);
        }

        $forms = $this->formGroupService->getFormsByFilters($departmentIds, $formName, $limit);

        return $this->apiResponse(
            200,
            'Forms retrieved successfully',
            null,
            new FormsCollection($forms)
        );
    }
}
